/*
    MultiChannelRelay.h
    Seeed multi channel relay library

    Copyright (c) 2018 Seeed Technology Co., Ltd.
    Author        :   lambor
    Create Time   :   June 2018
    Change Log    :

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "MultiChannelRelay.h"

MultiChannelRelay::MultiChannelRelay(I2C *i2c, uint8_t slave_adr)
    : _i2c(i2c)
    //, _i2cAddr(slave_adr)
{
    channel_state = 0;
    _i2cAddr = slave_adr << 1;
}


uint8_t MultiChannelRelay::getFirmwareVersion(void)
{
    char cmd[2];
    cmd[0] = CMD_READ_FIRMWARE_VER;
    
    sendToDevice( cmd, 1, 1, 10 );

    return cmd[0];
}

void MultiChannelRelay::changeI2CAddress(uint8_t old_addr, uint8_t new_addr)
{
    char cmd[2];
    cmd[0] = CMD_SAVE_I2C_ADDR;
    cmd[1] = new_addr;
    
    sendToDevice( cmd, 2, 0, 10 );
    _i2cAddr = new_addr;
}

uint8_t MultiChannelRelay::getChannelState(void)
{
    return channel_state;
}

void MultiChannelRelay::channelCtrl(uint8_t state)
{
    channel_state = state;
    char cmd[2];
    cmd[0] = CMD_CHANNEL_CTRL;
    cmd[1] = channel_state;
    
    sendToDevice( cmd, 2, 0, 10 );
}

void MultiChannelRelay::turn_on_channel(uint8_t channel)
{
    channel_state |= (1 << (channel - 1));
    char cmd[2];
    cmd[0] = CMD_CHANNEL_CTRL;
    cmd[1] = channel_state;
    
    sendToDevice( cmd, 2, 0, 10 );
}

void MultiChannelRelay::turn_off_channel(uint8_t channel)
{
    channel_state &= ~(1 << (channel - 1));
    char cmd[2];
    cmd[0] = CMD_CHANNEL_CTRL;
    cmd[1] = channel_state;
    
    sendToDevice( cmd, 2, 0, 10 );
}


/** Send a request to the chip, perform a wait and get response
 * @param cmd Pointer to character array containing request, no line endings needed
 * @param reqSize Size of the request in characters
 * @param respSize Size of the response in bytes, must be at least 1 character
 * @param msWait Delay between sending request and reading response in mS
 * @return true/false to indicate command was processed sucessfully or not, true success, false failure
 */
bool MultiChannelRelay::sendToDevice( char *cmd, int reqSize, int respSize, long msWait )
{
    bool retstat = false;
    int result = -10;
    _i2c->frequency(100000);
    _i2c->lock();
    result = _i2c->write(_i2cAddr, cmd, reqSize);
    if ( result ) {
        _i2c->unlock();
        return retstat;
    }

    wait_us( msWait * 100  );
    if( respSize == 0 ) {
        _i2c->unlock();
        return true;
    }

    result = _i2c->read(_i2cAddr, cmd, respSize);
    _i2c->unlock();
    if ( result ) {
    } else
        retstat = true;

    return retstat;
}


