/** 
 * LLAP Serial for use with Ciseco SRF/XRF wireless modules
 *
 * @author Andrew Lindsay
 *
 * @section DESCRIPTION
 *
 * Wireless modules available at http://shop.ciseco.co.uk/rf-module-range/
 * Library developped with ST Micro Nucleo F401 and Ciseco SRF shield.
 * Sheild needs to be modified as Tx/Rx on pins 0 and 1 conflict with the USB debug port.
 * They need joining to Rx - PA_12, Tx - PA_11 on the outer row of pins. 
 * See http://mbed.org/platforms/ST-Nucleo-F401RE/ for pinouts.
 *
 * This code is based on the Ciseco LLAPSerial library for Arduino at https://github.com/CisecoPlc/LLAPSerial but updated to
 * work as a mbed library
 *
 * Converted and updated by Andrew Lindsay @AndrewDLindsay April 2014
 *
 * For the SRF sheild to work correctly, some mods are needed to the Nucleo board. SB13 and SB14 need to be removed,
 * SB62 and SB63 need to be bridged. This then removes the UART from the USB output. An alternative UART is then needed
 * for debug purposes.
 * 
 * @section LICENSE
 *
 * The MIT License (MIT)
 * 
 * Copyright (c) 2014 Andrew Lindsay (andrew [at] thiseldo [dot] co [dot] uk)
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.

 */
 
#include "mbed.h"
#include "LLAPSerial.h"

// Constructors to pass in Tx/Rx, enable pins and optional ID, default is -- as defined in LLAPSerial.h
LLAPSerial::LLAPSerial(PinName txPin, PinName rxPin, PinName enableSRF, bool checkDevIDin, char *dID) : srf(txPin, rxPin), srfEnable( enableSRF )
{
    srf.baud(115200);
    
    bMsgReceived = false;
    setDeviceId(dID);
    cMessage[12]=0;     // ensure terminated
    inPtr = cMessage;
    checkDevID = checkDevIDin;
    
    // Enable the SRF
    srfEnable = 1;

    // Attach the receive interrupt handler
    srf.attach( this,&LLAPSerial::SerialEvent );
}


void LLAPSerial::processMessage()
{
    if( checkDevID ) {
        if (cMessage[1] != deviceId[0]) return;
        if (cMessage[2] != deviceId[1]) return;
    }
    // now we have LLAP.cMessage[3] to LLAP.cMessage[11] as the actual message
    if (0 == strncmp(&cMessage[3],"HELLO----",9)) {
        srf.printf("%s",cMessage); // echo the message
        return;
    } else if (0 == strncmp(&cMessage[3],"CHDEVID",7)) {
        if (strchr("-#@?\\*ABCDEFGHIJKLMNOPQRSTUVWXYZ", cMessage[10]) != 0 && strchr("-#@?\\*ABCDEFGHIJKLMNOPQRSTUVWXYZ", cMessage[11]) != 0) {
            deviceId[0] = cMessage[10];
            deviceId[1] = cMessage[11];
            srf.printf( "%s", cMessage); // echo the message
        }
    } else {
        strncpy(sMessage, &cMessage[0], 12); // let the main program deal with it
        bMsgReceived = true;
    }
}

void LLAPSerial::SerialEvent( void )
{
    if (bMsgReceived) return; // get out if previous message not yet processed
    if (srf.readable() ) {
        // get the new char:
        char inChar = (char)srf.getc();
        if (inChar == 'a') {
            // Start of a new message
            inPtr = cMessage;
        }
        *inPtr++ = inChar;
        if( inPtr >= &cMessage[12]) {
            // Message received, terminate, process and reset pointer
            *inPtr = '\0';
            processMessage();
            inPtr = cMessage;
        }
    }
}

/*
void LLAPSerial::SerialEvent( void )
{
    if (bMsgReceived) return; // get out if previous message not yet processed
    if (srf.readable() ) {
        // get the new byte:
        char inChar = (char)srf.getc();
        if (inChar == 'a') {
            cMessage[0] = inChar;
            for (int i = 1; i<12; i++) {
                inChar = (char)srf.getc();
                if( inChar == 'a' )
                    return;     // out of sync so abort and pick it up next time round
                cMessage[i] = inChar;
            }
            cMessage[12]=0;
            processMessage();
        } else
            srf.getc();  // throw away the character
    }
}
*/

void LLAPSerial::sendMessage(char *sToSend)
{
    cMessage[0] = 'a';
    cMessage[1] = deviceId[0];
    cMessage[2] = deviceId[1];
    for (int i = 0; i<9; i++) {
        if (i < strlen(sToSend) )
            cMessage[i+3] = sToSend[i];
        else
            cMessage[i+3] = '-';
    }

    srf.printf("%s",cMessage);
}

void LLAPSerial::sendMessage(char* sToSend, char* valueToSend)
{
    cMessage[0] = 'a';
    cMessage[1] = deviceId[0];
    cMessage[2] = deviceId[1];
    for (int i = 0; i<9; i++) {
        if (i < strlen(sToSend))
            cMessage[i+3] = sToSend[i];
        else if (i < strlen(sToSend) + strlen(valueToSend))
            cMessage[i+3] = valueToSend[i - strlen(sToSend)];
        else
            cMessage[i+3] = '-';
    }

    srf.printf("%s", cMessage);
}


void LLAPSerial::sendInt(char *sToSend, int value)
{
    char cValue[7];     // long enough for -32767 and the trailing zero
    sprintf( cValue,"%d", value );
    int cValuePtr = 0;

    cMessage[0] = 'a';
    cMessage[1] = deviceId[0];
    cMessage[2] = deviceId[1];
    for (int i = 0; i<9; i++) {
        if (i < strlen(sToSend))
            cMessage[i+3] = sToSend[i];
        else if (cValuePtr < 7 && cValue[cValuePtr] !=0)
            cMessage[i+3] = cValue[cValuePtr++];
        else
            cMessage[i+3] = '-';
    }

    srf.printf("%s",cMessage);
}

void LLAPSerial::sendIntWithDP(char *sToSend, int value, int decimalPlaces)
{
    char cValue[8];     // long enough for -3276.7 and the trailing zero
    int cValuePtr=0;
    //itoa(value, cValue,10);
    sprintf( cValue,"%d", value );
    char* cp = &cValue[strlen(cValue)];
    *(cp+1) = 0;    // new terminator
    while (decimalPlaces-- && --cp ) {
        *(cp+1) = *cp;
    }
    *cp = '.';

    cMessage[0] = 'a';
    cMessage[1] = deviceId[0];
    cMessage[2] = deviceId[1];
    for (int i = 0; i<9; i++) {
        if (i < strlen(sToSend))
            cMessage[i+3] = sToSend[i];
        else if (cValuePtr < 8 && cValue[cValuePtr] !=0)
            cMessage[i+3] = cValue[cValuePtr++];
        else
            cMessage[i+3] = '-';
    }

    srf.printf("%s", cMessage);
}

void LLAPSerial::setDeviceId(char* cId)
{
    deviceId[0] = cId[0];
    deviceId[1] = cId[1];
}
