/** LLAPSerial.h
 * @author Andrew Lindsay
 * 
 * @section LICENSE
 *
 * The MIT License (MIT)
 * 
 * Copyright (c) 2014 Andrew Lindsay
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef _LLAPSERIAL_h
#define _LLAPSERIAL_h

#include "mbed.h"

/** LLAP Serial class for communicating with Ciseco wireless devices
 *
 */
class LLAPSerial
{
 private:
    /** Private internal received message buffer used when receivign characters
    */
    char cMessage[13];      // Raw receive buffer
    
    /** Internal pointer to next character to write in buffer
    */
    char* inPtr;
    
    /** Boolean flag to determine if code needs to check the incoming ID and match to device ID, if true and no match,
        message is ignored
    */
    bool checkDevID;
    
    /** Process a received message, determine if its for us, starts with 'a' and optionally matches device ID.
        Automatically responds to HELLO and CHDEVID requests (TODO - Add more auto responses)
    */
    void processMessage();
    
    /** Serial interrupt handler, deals with a single character, builds up internal receive message buffer and calls processmessage when
        enough characters have been received.
    */
    void SerialEvent();
    
    /** Serial object to communicate with the SRF/XRF
    */
    Serial srf;
    
    /** Pin for SRF enable/disable
    */
    DigitalOut srfEnable;
    
 public:
     /** Default Constructor
     * @param txPin Pin name for UART TX
     * @param rxPin Pin name for UART RX
     * @param enableSRF Pin name for the SRF enable
     * @param checkNodeID Check the incoming Device ID and ignore if it doesnt match
     * @param dID ID for device, used in sending messages and if checkNodeID is true then only messages for this ID are accepted
     */
    LLAPSerial(PinName txPin, PinName rxPin, PinName enableSRF, bool checkDevIDin = false, char *dID = "--" );

    /** Send a message
    * @param sToSend Character pointer to message to send, will be padded with - to full length or truncated if too long
    */
    void sendMessage(char* sToSend);

    /** Send a message with a value. Both are concatenated in the message with padding - if needed
    * @param sToSend Character pointer to message to send
    * @param valueToSend The value to send as a series of characters
    */
    void sendMessage(char* sToSend, char* valueToSend);

    /** Send an integer
    * @param sToSend Character pointer to message to send
    * @param value The value to send as a series of characters, will be padded with - to full length or truncated if too long
    */
    void sendInt(char *sToSend, int value);

    /** Send integer message to specified decimal places
    * @param sToSend Character pointer to message to send
    * @param value The value to send as a series of characters, will be padded with - to full length or truncated if too long
    * @param decimalPlaces The number of decimal places to use.
    */
    void sendIntWithDP(char *sToSend, int value, int decimalPlaces);

    /** Set device ID
    * @param cId New device ID to set
    */
    void setDeviceId(char* cId);

    // Public variables for message processing by main application (ideally should be getter functions)    

    /** ID of this device
    */
    char deviceId[2];

    /** Public Received message buffer
    */
    char sMessage[15];      // Received message buffer

    /** Flag to indicate a new message has been received, must be cleared after message processed
    */
    bool bMsgReceived;
};

#endif

