/** IoT Gateway RFM12B payload V2 format handling
 *
 * @author Andrew Lindsay
 *
 * @section LICENSE
 *
 * Copyright (c) 2012 Andrew Lindsay (andrew [at] thiseldo [dot] co [dot] uk)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * 
 * @section DESCRIPTION
 * 
 * 
 */
 
#ifndef _PAYLOADV2_H
#define _PAYLOADV2_H

#include "mbed.h"
#include "PayloadDef.h"

#define V2_DATATYPE_BYTE    0
#define V2_DATATYPE_SHORT   1
#define V2_DATATYPE_LONG    2
#define V2_DATATYPE_STRING  3

/** V2 header definition
 */
typedef struct {
    uint8_t typeId;        // Type and sensor ID high
    uint8_t id;            // Sensor ID
}  __attribute__((packed)) payloadv2Header;

/** Byte reading definition
 */
typedef struct {
    uint16_t type:4;        // Type
    uint16_t id:12;         // Sensor ID
    int8_t reading;         // Reading
}  __attribute__((packed)) payloadv2ByteReading;

/** Short reading definition
 */
typedef struct {
    uint16_t type:4;        // Type
    uint16_t id:12;         // Sensor ID
    int16_t reading;        // Reading
}  __attribute__((packed)) payloadv2ShortReading;

/** Long reading definition
 */
typedef struct {
    uint16_t type:4;        // Type
    uint16_t id:12;         // Sensor ID
    int32_t reading;        // Reading
}  __attribute__((packed)) payloadv2LongReading;

/** String reading definition
 */
typedef struct {
    uint16_t type:4;        // Type
    uint16_t id:12;         // Sensor ID
    int8_t length;          // String length
    uint8_t str[1];         // First character or data
}  __attribute__((packed)) payloadv2StringReading;

/** Class definition for Payload V2 format
 */
class PayloadV2: public PayloadDef {
public:
    /** Default Constructor
     */
    PayloadV2();

    /** Alternative constructor for passing payload details
     *
     * @param pptr  Pointer to payload data
     * @param plen  Length of paylod data
     */
    PayloadV2( uint8_t *pptr, short plen );

    /** Get the number of readings in a payload
     *
     * @returns the number of readings found in payload so they can be read
     */
    virtual short numReadings();

    /** Get a single reading from the payload
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns Integer representation of reading
     */
    virtual int reading( short readingNum );

    /** Get the sensor ID for the specified reading within the payload
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns Integer representation of reading
     */
    virtual short sensorId( short readingNum );

    /* The next functions are extra to the PayloadV2 type */
    
    /** Get the pointer to the specified reading within the payload
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns Pointer to the reading
     */
    uint8_t *readingPtr( short readingNum );

    /** Get a single Byte reading from the payload
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns Byte 8 bit Integer representation of reading
     */
    int8_t readingByte( short readingNum );
    
    /** Get a single short reading from the payload
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns Short, 16 bit Integer representation of reading
     */
    short readingShort( short readingNum );
    
    /** Get a single Long reading from the payload
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns Long 32bit Integer representation of reading
     */
    int readingLong( short readingNum );

    /** Get the type of the specified reading
     *
     * @param readingNum The number of the reading, starts from 0 upto number of readings in payload
     * @returns type of the reading, byte, short, long, string.
     */
    short readingType( short readingNum );
};

#endif /* _PAYLOADV2_H */
