/** IoT Gateway Routing for incoming messages
 *
 * @author Andrew Lindsay
 *
 * @section LICENSE
 *
 * Copyright (c) 2012 Andrew Lindsay (andrew [at] thiseldo [dot] co [dot] uk)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * 
 * @section DESCRIPTION
 * 
 * 
 */

#ifndef _IOTROUTING_H
#define _IOTROUTING_H

#include "iotgateway.h"
#include "mbed.h"
#include "HTTPClient.h"
#include "OutputDef.h"
#include "OutputPachube.h"
#include "OutputMqtt.h"
#include "OutputEmonCms.h"
#include "OutputSenSe.h"
#include "PayloadV1.h"
#include "PayloadV2.h"
#include "PayloadSimple.h"
#include <vector>

#define STATUS_PAYLOAD_VMASK 0xf8
#define STATUS_PAYLOAD_VERSION 0x06
#define STATUS_PAYLOAD_V1      0x00
#define STATUS_PAYLOAD_V2      0x02
#define STATUS_LOW_BATTERY     0x01

#define PAYLOAD_TYPE_UNKNOWN 99
#define PAYLOAD_TYPE_SIMPLE 0
#define PAYLOAD_TYPE_V1 1
#define PAYLOAD_TYPE_V2 2

#define OUTPUT_UNKNOWN 0
#define OUTPUT_TYPE_PACHUBE 1
#define OUTPUT_TYPE_MQTT 2
#define OUTPUT_TYPE_EMONCMS 3
#define OUTPUT_TYPE_SENSE 4

/** Routing definition for a reading
 * Input source, node ID, sensor ID, output
 * default is OuptutDef but casts to Pachube or MQTT
 */
typedef struct {
    short nodeId;
    short sensorId;
    float factor;
    byte outType;
    OutputDef *output;
    char *param1;
    char *param2;
    char *param3;
    char *param4;
}  __attribute__((packed)) PayloadRouting;

/** Node definition
 */
typedef struct {
    uint8_t groupId;
    uint8_t nodeId;
    uint8_t length;
    uint8_t type;
}  __attribute__((packed)) IoTNodeDef;


/** Main IoT Gateway RFM12B payload routing class
 */
class IoTRouting {
public:
    /** Default Constructor
     */
    IoTRouting();

    /** Initialise routing class
     */
    void initRouting();
    
    /** Write a list of nodes to IOTNODE.TXT
     * Format is: group,node,length,type
     * Where:
     *   group, node, length are first 3 bytes of received packet
     *   type is 99 for unknown, 0 for simple, 1 for Type 1, 2 for Type 2
     */
    bool writeNodeList();
    
    /** Write main routing list to file IOTRTR.TXT
     * Format is: nodeId,sensorId,factor,outputType,param1,param2,param3,param4
     * Where:
     *   nodeId is the Node ID of the node to the sensor is attached to
     *   sensorId is the ID of the sensor to define
     *   factor is a conversion factor to convert from integer to real value. can be 100.0,10.0,1.0,0.1,0.01,0.001. For no conversion use 1.0
     *   outputType is the type of output, 1 is Pachube, 2 is MQTT
     *   param1 is Pachube feed number or MQTT topic
     *   param2 is Pachube datastream number
     *   param3 and 4 are currently unused 
     */
    bool writeRoutingList();

    /** Add a node to the nodelist
     *
     * @param groupId Group ID, byte 0 of received packet
     * @param nodeId  Node ID, byte 1
     * @param length  Length of payload, byte 2
     * @param type    Type of payload, simple, v1, v2 or undefined
     */
    void addNodeToList(uint8_t groupId, uint8_t nodeId, uint8_t length, uint8_t type );

    /** Add an item to routing list
     *
     * @param nodeId   ID of the node payload came from
     * @param sensorId ID of the sensor within the node
     * @param factor   Sensor Reading conversion factor, 100, 10, 1, 0.1, 0.01, 0.001 etc
     * @param outType  Output type, 1 = Pachube, 2 = MQTT, 3 = emonCMS, 4 = Sen.Se
     * @param param1   Parameter 1
     * @param param2   Parameter 2
     * @param param3   Parameter 3
     * @param param4   Parameter 4
     */
    void addRoutingToList( short nodeId, short sensorId, float factor, byte outType,
            char *param1, char *param2, char *param3, char *param4 );
            
    /** get Payload type 
     *
     * @param data    Pointer to start of payload data
     * @param dataLen Length of payload data
     * @returns payload type, either simple, v1, v2 or unknown
     */
    short getPayloadType(  uint8_t *data, short dataLen );
    
    /** Get routing data for specified node/sensor combination
     *
     * @param nodeId    The ID of the node
     * @param sensorNum The ID of the sensor 
     * @returns pointer to routing data
     */
    PayloadRouting* getRouting( short nodeId, short sensorNum );
    
    /** Add output definition to output list
     *
     * @param output  Pointer to output object
     * @param outType Numeric value for the output type
     */
    void addOutput( OutputDef *output, short outType );

    /** Main payload routing function
     *
     * @param payload     Pointer to payload data
     * @param payloadLen  Length of th epayload
     * @returns flag to indicate payload routed to output or not.
     */  
    bool routePayload( uint8_t *payload, short payloadLen );
    
private:
    int sendCount;
    OutputDef *pachubeOutput;
    OutputDef *mqttOutput;
    OutputDef *emonCmsOutput;
    OutputDef *senSeOutput;

    std::vector<PayloadRouting*> _routing;
    std::vector<IoTNodeDef*> _nodeDefList;

};

#endif /* _IOTROUTING_H */
