/** Library for Holtek HT1632 LED driver chip,
 * As implemented on the Sure Electronics DE-DP10X or and DE-DP13X1X display board
 * 8 x 32 dot matrix LED module.)
 *
 * Original code by:
 * Nov, 2008 by Bill Westfield ("WestfW")
 *   Copyrighted and distributed under the terms of the Berkely license
 *   (copy freely, but include this notice of original author.)
 *
 * Adapted for 8x32 display by FlorinC.
 *
 * Arduino Library Created and updated by Andrew Lindsay October/November 2009
 *
 * Ported to Mbed platform by Andrew Lindsay, November 2012
 *
 * @author Andrew Lindsay
 *
 * @section LICENSE
 *
 * Copyright (c) 2012 Andrew Lindsay (andrew [at] thiseldo [dot] co [dot] uk)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * 
 * @section DESCRIPTION
 *  Definitions for Holtek HT1632 LED driver
 * 
 */

#ifndef _HT1632_LEDMATRIX_H
#define _HT1632_LEDMATRIX_H

#include "mbed.h"
#include "inttypes.h"

// To include the graphic functions use the following. Comment out to exclude them 
#undef USE_GRAPHIC

// Defines for display sizes
#define HT1632_08x32  1 
#define HT1632_16x24  2

/*
 * commands written to the chip consist of a 3 bit "ID", followed by
 * either 9 bits of "Command code" or 7 bits of address + 4 bits of data.
 */
#define HT1632_ID_CMD 4     /* ID = 100 - Commands */
#define HT1632_ID_RD  6     /* ID = 110 - Read RAM */
#define HT1632_ID_WR  5     /* ID = 101 - Write RAM */

#define HT1632_CMD_SYSDIS 0x00  /* CMD= 0000-0000-x Turn off oscil */
#define HT1632_CMD_SYSON  0x01  /* CMD= 0000-0001-x Enable system oscil */
#define HT1632_CMD_LEDOFF 0x02  /* CMD= 0000-0010-x LED duty cycle gen off */
#define HT1632_CMD_LEDON  0x03  /* CMD= 0000-0011-x LEDs ON */
#define HT1632_CMD_BLOFF  0x08  /* CMD= 0000-1000-x Blink ON */
#define HT1632_CMD_BLON   0x09  /* CMD= 0000-1001-x Blink Off */
#define HT1632_CMD_SLVMD  0x10  /* CMD= 0001-00xx-x Slave Mode */
#define HT1632_CMD_MSTMD  0x14  /* CMD= 0001-01xx-x Master Mode */
#define HT1632_CMD_RCCLK  0x18  /* CMD= 0001-10xx-x Use on-chip clock */
#define HT1632_CMD_EXTCLK 0x1C  /* CMD= 0001-11xx-x Use external clock */
#define HT1632_CMD_COMS00 0x20  /* CMD= 0010-ABxx-x commons options */
#define HT1632_CMD_COMS01 0x24  /* CMD= 0010-ABxx-x commons options */
#define HT1632_CMD_COMS10 0x28  /* CMD= 0010-ABxx-x commons options */
#define HT1632_CMD_COMS11 0x2C  /* CMD= 0010-ABxx-x commons options */
#define HT1632_CMD_PWM    0xA0  /* CMD= 101x-PPPP-x PWM duty cycle */

#define PIXEL_OFF 0
#define PIXEL_ON  1

/** class HT1632_LedMatrix
*/
class HT1632_LedMatrix : public Stream
{
public:
    /** Default Constructor for the display driver
     */
//    HT1632_LedMatrix( );
    HT1632_LedMatrix( PinName clk, PinName dat, PinName cs1, PinName cs2 , PinName cs3, PinName cs4);


    /** Initialise the library with the display configuration
     * @param Number of horizontal displays, max 4
     * @param Number of vertical displays, max 4
     * @param Type of displays being used, either 32x8 or 24x16
     */
    void init( uint8_t, uint8_t, uint8_t displayType = HT1632_08x32 );
    
    /** Turn off the display
     */
    void displayOff( void );
    
    /** Turn on the display
     */
    void displayOn( void );
    
    /** Clear the display, and the shadow memory, and the snapshot
     * memory.  This uses the "write multiple words" capability of
     * the chipset by writing all 96 words of memory without raising
     * the chipselect signal.
     */
    void clear(void);
    
    /** Set display brightness. Brighness is from 0 to 15
     * @param brightness value, range x-y
     */
    void setBrightness( unsigned char );
    
    /** Copy a character glyph from the font data structure to
     * display memory, with its upper left at the given coordinate
     * This is unoptimized and simply uses plot() to draw each dot.
     * returns number of columns that didn't fit
     * @param column
     * @param row
     * @param character to write
     * @returns the number of columns that couldnt be displayed on current matrix
     */
    uint8_t putChar( int, int, char );

    /** Copy a character glyph from the font data structure to
     * display memory, with its upper left at the given coordinate
     * This is a wrapper for putChar that uses current cursor position
     */
    void write( uint8_t );

    /** Write a string at the position specified
     * x and y start from 0 and count number of pixels, 2nd row on a 2 row display is y=8
     * @param x position
     * @param y position
     * @param pointer to character data to display
     */
    void putString( int, int, char* );
    
    /** Plot a point on the display, with the upper left hand corner
     * being (0,0), and the lower right hand corner being (xMax-1, yMax-1).
     * Note that Y increases going "downward" in contrast with most
     * mathematical coordiate systems, but in common with many displays
     * basic bounds checking used.
     * @param column
     * @param row
     * @param point on or off
     */
    void plot( int, int, char );
    
    /** Set cursor position
     * @param x or column position, 0 is to left
     * @param y or row position, 0 is at top
     */
    void gotoXY(int , int);
    
    /** Return the current cursor position
     * @param column pointer
     * @param row pointer
     */
    void getXY(int* , int*);

    /** Return the maximum size of the display
     * @param column pointer
     * @param row pointer
     */
    void getXYMax(int*, int*);
    
    /** Shift cursor X position a number of positions either left or right.
     * @param increment, -1 for one position left, 1 for right
     */
    void shiftCursorX(int );
    
    /** Create a custom character. 8 character slots are availabe.
     * @param character number, 0 to 7
     * @param array of character bit maps
     */
    void setCustomChar( int, unsigned char[]);
    
    /** Create a custom character. 8 character slots are availabe.
     * @param character number, 0 to 7
     * @param array of character bit maps
     * @param width of chaaracter in columns
     */
    void setCustomChar( int, unsigned char[], uint8_t );
    
    /** Scroll row to the left
     * @param number fo columns to scroll
     * @param row number to scroll, used when displays are stacked vertically or when more than 8 high displays are used
     */
    void scrollLeft(uint8_t, uint8_t);
    
    /** Write shaddow ram to display
     */
    void putShadowRam();
    
    /** Write shadow ram to a specific dasipaly board
     * @param chip or display number
     */
    void putShadowRam(uint8_t);
    // Graphic functions
#ifdef  USE_GRAPHIC

    /** Draws a line between two points on the display
     * @param start column
     * @param start row
     * @param end column
     * @param end row
     * @param point on or off, off to erase, on to draw
     */
    void drawLine(unsigned char x1, unsigned char y1,
                  unsigned char x2, unsigned char y2, unsigned char c);

    /** Draw a rectangle given to top left and bottom right points
     * @param top left column
     * @param top left row
     * @param bottom right column
     * @param bottom right row
     * @param point on or off, off to erase, on to draw
     */                  
    void drawRectangle(unsigned char x1, unsigned char y1,
                       unsigned char x2, unsigned char y2, unsigned char c);
    /** Draw a filled rectangle given to top left and bottom right points
     * @param top left column
     * @param top left row
     * @param bottom right column
     * @param bottom right row
     * @param point on or off, off to erase, on to draw
     */                  
    void drawFilledRectangle(unsigned char x1, unsigned char y1,
                             unsigned char x2, unsigned char y2, unsigned char c);
    /** Draw a circle using Bresenham's algorithm.
     * Some small circles will look like squares!!
     * @param centre column
     * @param centre row
     * @param radius
     * @param point on or off, off to erase, on to draw
     */
    void drawCircle(unsigned char xc, unsigned char yc,
                    unsigned char r, unsigned char c);
#endif

private:
    /** Select a specific display, numbers as 0 to 3
     *
     * @param chip number
     */
    void chipselect( uint8_t );
    /** Deselect a specific display, numbers as 0 to 3
     *
     * @param chip number
     */
    void chipfree( uint8_t );
    
    /** Write bits to HT1632 on pins HT1632_DATA, HT1632_WRCLK
     * Chip is assumed to already be chip-selected
     * Bits are shifted out from MSB to LSB, with the first bit sent
     * being (bits & firstbit), shifted till firsbit is zero.
     * @param bits
     * @param firstbit
     */ 
    void writebits( uint8_t, uint8_t );
    
    /** Write databits to HT1632 on pins HT1632_DATA, HT1632_WRCLK
     * Chip is assumed to already be chip-selected
     * Bits are shifted out from LSB to MSB
     * @param bits
     * @param count
     */
    void writedatabits( uint8_t, uint8_t );
    
    /**  * Send a command to the ht1632 chip.
     * A command consists of a 3-bit "CMD" ID, an 8bit command, and
     * one "don't care bit".
     *   Select 1 0 0 c7 c6 c5 c4 c3 c2 c1 c0 xx Free
     * @param chipno the number of the chip/display to write command to
     * @param command to send
     */
    void sendcmd( uint8_t, uint8_t );
    
    /** Send a nibble (4 bits) of data to a particular memory location of the
     * ht1632.  The command has 3 bit ID, 7 bits of address, and 4 bits of data.
     *    Select 1 0 1 A6 A5 A4 A3 A2 A1 A0 D0 D1 D2 D3 Free
     * Note that the address is sent MSB first, while the data is sent LSB first!
     * This means that somewhere a bit reversal will have to be done to get
     * zero-based addressing of words and dots within words.
     * @param chipno
     * @param address
     * @param data
     */
    void senddata( uint8_t, uint8_t, uint8_t );

    /** Send a byte of data to a particular memory location of the
     * ht1632.  The command has 3 bit ID, 7 bits of address, and 8 bits of data.
     *    Select 1 0 1 A6 A5 A4 A3 A2 A1 A0 D0 D1 D2 D3 D4 D5 D6 D7 D8 Free
     * Note that the address is sent MSB first, while the data is sent LSB first!
     * This means that somewhere a bit reversal will have to be done to get
     * zero-based addressing of words and dots within words.
     * @param chipno
     * @param address
     * @param data
     */
    void sendcol( uint8_t, uint8_t, uint8_t );
    
    DigitalOut _wrclk;
    DigitalOut _data;
    DigitalOut _cs1;
    DigitalOut _cs2;
    DigitalOut _cs3;
    DigitalOut _cs4;
    
//protected:
    virtual int _putc(int value);
    virtual int _getc();

};

#endif //_HT1632_LEDMATRIX_H
