#include "ble/BLE.h"
#include "mbed.h"
#include "SmartPlugService.h"
#include "SmartPlugBLE.h"



void convert(uint8_t* res,uint32_t &num)
{
    res[3] = (num&0xFF000000)>>24;
    res[2] = (num&0x00FF0000)>>16;
    res[1] = (num&0x0000FF00)>>8;
    res[0] = num&0x000000FF;
}

SmartPlugService::SmartPlugService(BLE &_ble, SmartPlugBLE &sys):
        ble(_ble), led(LED3,1),system(sys),
        voltageChar(SPS_UUID_VOLTAGE_CHAR,&voltage,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        currentChar(SPS_UUID_CURERNT_CHAR,&current,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        powerChar(SPS_UUID_POWER_CHAR,&power,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        powerFactorChar(SPS_UUID_POWER_FACTOR_CHAR,&powerFactor,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        energyChar(SPS_UUID_ENERGY_CHAR,&energy,GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        relayChar(SPS_UUID_RELAY_CHAR, relayValue.getDataPointer(),
                relayValue.getLenBytes(), RelayValueBytes::MAX_SIZE_BYTES,
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ),
        updateChar(SPS_UUID_UPDATE_CHAR,NULL)
{
    system.addObserver(this);
}

void SmartPlugService::onDataWritten(const GattWriteCallbackParams *params) 
{
    if(params->handle == updateChar.getValueHandle())
    {
        system.updateData();
    }
    else if(params->handle == relayChar.getValueHandle())
    {
            system.onRelayWrite(params->data);
    }
}

void SmartPlugService::updateObserver(void* data)
{
    SmartPlug* sp = (SmartPlug*)data;
    updateVoltage(sp->getVoltage());
    updateCurrent(sp->getCurrent());
    updatePower(sp->getPower());
    updatePowerFactor(sp->getPowerFactor());
    updateEnergy(sp->getEnergy());
    updateRelay(sp->getRelay());
}

void SmartPlugService::updateVoltage(uint32_t v)
{
    if (ble.getGapState().connected) 
    {
        convert(voltage,v);
        ble.updateCharacteristicValue(voltageChar.getValueHandle(),voltage,sizeof(voltage));
    }
}

void SmartPlugService::updateCurrent(uint32_t c)
{
    if (ble.getGapState().connected) 
    {
        convert(current,c);
        ble.updateCharacteristicValue(currentChar.getValueHandle(),(current),sizeof(current));
    }
}

void SmartPlugService::updatePower(uint32_t p)
{
    if (ble.getGapState().connected) 
    {
        convert(power,p);
        ble.updateCharacteristicValue(powerChar.getValueHandle(),(power),sizeof(power));        
    }
}

void SmartPlugService::updatePowerFactor(uint32_t pf)
{
    if (ble.getGapState().connected) 
    {    
        convert(powerFactor,pf);
        ble.updateCharacteristicValue(powerFactorChar.getValueHandle(),(powerFactor),sizeof(powerFactor));
    }
}

void SmartPlugService::updateEnergy(uint32_t e)
{
    if(ble.getGapState().connected)
    {
        convert(energy,e);
        ble.updateCharacteristicValue(energyChar.getValueHandle(),energy,sizeof(energy));
    }
}

void SmartPlugService::updateRelay(Relay* relay)
{
    if (ble.getGapState().connected) 
    {   
        relayValue.updateData(relay);
        ble.updateCharacteristicValue(relayChar.getValueHandle(),relayValue.getDataPointer(),
        relayValue.getLenBytes());
    }
}

void SmartPlugService::setupService(void) 
{

    static bool serviceAdded = false; /* We should only ever need to add the service once. */
    if (serviceAdded)
    {
        return;
    }

        
    GattCharacteristic *charTable[] = {&voltageChar,&currentChar,&powerChar,
                                        &powerFactorChar,&energyChar,&relayChar,
                                        &updateChar
                                        };
    GattService smartPlugService = GattService(SPS_UUID_SERVICE, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
    ble.addService(smartPlugService);

}
