#ifndef S25FL216K_H
#define S25FL216K_H

#include "mbed.h"

#define S25FL_WRITE_STATUS      0x01
#define S25FL216K_WRITE         0x02
#define S25FL216K_READ          0x03
#define S25FL216K_RDSR          0x05
#define S25FL216K_WREN          0x06
#define S25FL216K_SECTOR_ERASE  0x20
#define S25FL216K_CHIP_ERASE    0xC7
#define S25FL216K_BLOCK_ERASE   0xD8
#define S25FL216K_JEDEC         0x9F

/**
* Class to write/read from S25FL216K serial flash memory
*
* This class is used for binary access, for file system access
* there is S25FL216K_USBFileSystem.
*
* No write restrictions are implemented, the memory also should not
* have write restrictions enabled, this is factory default.
*/
class S25FL216K {
public:

/** Constructor
*
* @param mosi - Mosi pin to be used
* @param miso - Miso pin to be used
* @param sclk - Sclk pin to be used
* @param cs - CS pin to be used
*/
S25FL216K(PinName mosi, PinName miso, PinName sclk, PinName cs);

/** Checks if communication with the device functions
*
* @return - true upon succesfull communication, false if it failed
*/
bool testConnection( void );

/** Read a number of memory locations
*
* Take into account reading will stay within the page (256B) the address is in
*
* @param address - start address to read from
* @param buffer - char array to read the data into
* @param length - number of samples to read
*/
void read(int address, char *buffer, int length);

/** Write a number of memory locations
*
* Take into account writing will stay within the page (256B) the address is in
*
* @param address - start address to write to
* @param buffer - char array to read the data from
* @param length - number of samples to write
* @param block - true (default) to block until writing is finished 
*/
void write(int address, char *buffer, int length, bool block = true);

/** Erase a sector (4KB)
*
* @param address - address in the sector that needs to be erased 
* @param block - true (default) to block until erasing is finished 
*/
void eraseSector(int address, bool block = true);

/** Erase a block (64KB)
*
* @param address - address in the block that needs to be erased 
* @param block - true (default) to block until erasing is finished 
*/
void eraseBlock(int address, bool block = true);

/** Erase the entire chip
* 
* @param block - true (default) to block until erasing is finished 
*/
void eraseChip(bool block = true);

/** Check if writing/erasing is being performed
*
* @return - true if busy, false if idle
*/
bool isBusy( void );

private:

char getStatus( void );
void setWriteEnable( void );

SPI _spi;
DigitalOut _cs;





};



#endif