#include "mbed.h"
#include "FastIO.h"
 
DigitalInOut led1(LED1);
FastInOut<LED2> led2;
 
Timer t;
#define LOOPS 10000

void basic_test();
void fixed_write();
void variable_write();
void read();
void operator_toggle();
void input_output();

void print_results(int digicount, int fastcount);

int main() {
    printf("\r\nStarting test bench\r\n");
    
    basic_test();
    fixed_write();
    variable_write();
    read();
    operator_toggle();
    input_output();
    
    while(1);
}

void print_results(int digicount, int fastcount) {
    float digicycles = (float)digicount / LOOPS * (float)SystemCoreClock / 1000000.0f;
    float fastcycles = (float)fastcount / LOOPS * (float)SystemCoreClock / 1000000.0f;
    printf("Standard took %.2f cycles, FastIO took %.2f cycles, which is %d%%\r\n", digicycles, fastcycles,(int)(100.0f*fastcycles/digicycles + 0.5));
    printf("Standard took %.0fns, FastIO took %.0fns\r\n", (float)digicount/LOOPS * 1000, (float)fastcount/LOOPS * 1000);
}
 
void basic_test() {
printf("Verifying basic behavior\r\n");
    int error = 0;
    led1.output();
    led2.output();
    led1 = 1;
    led2 = 1;
    error += (led1.read() != 1);
    error += (led2.read() != 1);
    led1 = 0;
    led2 = 0;
    error += (led1.read() != 0);
    error += (led2.read() != 0);
    
    if (error == 0) 
        printf("Basic behavior verified\r\n");
    else {
        printf("Error in basic behavior\r\n");
        while(1);
    }
}


void fixed_write() {
    int overhead;
    int digitalinout;
    int fastinout;
    int count;
    
    printf("\nMeasuring fixed write pattern speed\r\n");
    led1.output();
    led2.output();
    
    //Calculate loop overhead
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led1.write(1);
        led1.write(0);
    }
    t.stop();
    overhead = t.read_us();
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- )
    {
        led1.write(1);
        led1.write(0);
        led1.write(1);
        led1.write(0);
    }
    t.stop();
    digitalinout = t.read_us() - overhead;
    
    //Calculate loop overhead
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led2.write(1);
        led2.write(0);
    }
    t.stop();
    overhead = t.read_us();
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- )
    {
        led2.write(1);
        led2.write(0);
        led2.write(1);
        led2.write(0);
    }
    t.stop();
    fastinout = t.read_us() - overhead;
    
    print_results(digitalinout, fastinout);
}

void variable_write() {
    int overhead;
    int digitalinout;
    int fastinout;
    int count;
    
    printf("\nMeasuring variable write pattern speed\r\n");
    led1.output();
    led2.output();
    
    //Calculate loop overhead
    int value = 1;
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        value = value - 1; led1.write(value);
        value = value - 1; 
        value = value - 1;
        value = value - 1; led1.write(value);
    }
    
    t.stop();
    overhead = t.read_us();

    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- )
    {
        value = value - 1; led1.write(value);
        value = value - 1; led1.write(value);
        value = value - 1; led1.write(value);
        value = value - 1; led1.write(value);
    }
    t.stop();
    digitalinout = t.read_us() - overhead;
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        value = value - 1; led2.write(value);
        value = value - 1; 
        value = value - 1;
        value = value - 1; led2.write(value);
    }
    
    t.stop();
    overhead = t.read_us();

    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- )
    {
        value = value - 1; led2.write(value);
        value = value - 1; led2.write(value);
        value = value - 1; led2.write(value);
        value = value - 1; led2.write(value);
    }
    t.stop();
    fastinout = t.read_us() - overhead;
    print_results(digitalinout, fastinout);
}

void read() {
    int overhead;
    int digitalinout;
    int fastinout;
    int count;
    
    printf("\nMeasuring read speed\r\n");
    led1.input(); led1.mode(PullUp);
    led2.input(); led2.mode(PullUp);
    
    //Calculate loop overhead
    count = LOOPS;
    t.reset();
    t.start();
    while ( count -- ) {
    }
    t.stop();
    overhead = t.read_us();

    count = LOOPS;
    t.reset();
    t.start();
    while ( count -- )
    {
        if (led1.read() == 2)       //This shouldn't happen due to pull-up, if you get weird results, it is happening
            break;   
    }
    t.stop();
    digitalinout = t.read_us() - overhead;

    count = LOOPS;
    t.reset();
    t.start();
    while ( count -- )
    {
        if (led2.read() == 2)       //This shouldn't happen due to pull-up, if you get weird results, it is happening
            break;       
    }
    t.stop();
    fastinout = t.read_us() - overhead;
 
    print_results(digitalinout, fastinout);
}

void operator_toggle() {
    int overhead;
    int digitalinout;
    int fastinout;
    int count;
    
    printf("\nMeasuring toggling using operators speed\r\n");
    led1.output();
    led2.output();
    
    //Calculate loop overhead
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led1 = !led1;      
        led1 = !led1;
    }
    t.stop();
    overhead = t.read_us();

    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led1 = !led1;      
        led1 = !led1;
        led1 = !led1;      
        led1 = !led1;
    }
    t.stop();
    digitalinout = t.read_us() - overhead;
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led2 = !led2;
        led2 = !led2;
    }
    t.stop();
    overhead = t.read_us();
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- )
    {
         led2 = !led2;
         led2 = !led2;
         led2 = !led2;
         led2 = !led2;
    }
    t.stop();
    fastinout = t.read_us() - overhead;
    print_results(digitalinout, fastinout);
}

void input_output() {
    int overhead;
    int digitalinout;
    int fastinout;
    int count;
    
    printf("\nMeasuring switching between input and output\r\n");
    
    //Calculate loop overhead
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led1.input();
        led1.output();
    }
    t.stop();
    overhead = t.read_us();

    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led1.input();
        led1.output();
        led1.input();
        led1.output();
    }
    t.stop();
    digitalinout = t.read_us() - overhead;
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- ) {
        led2.input();
        led2.output();
    }
    t.stop();
    overhead = t.read_us();
    
    count = LOOPS / 2;
    t.reset();
    t.start();
    while ( count -- )
    {
        led2.input();
        led2.output();
        led2.input();
        led2.output();
    }
    t.stop();
    fastinout = t.read_us() - overhead;
    print_results(digitalinout, fastinout);
}