/* Library for AK8975 digital compass IC, for use with I2C 

This library contains basic functionality, if you want more, make it ;) 
         
*/

#ifndef AK8975_H
#define AK8975_H

/**
 * Includes
 */
#include "mbed.h"

/**
 * Registers
 */
#define AK8975_ID_REG       0x00
#define AK8975_ST1_REG      0x02
#define AK8975_X_REG        0x03
#define AK8975_Y_REG        0x05
#define AK8975_Z_REG        0x07
#define AK8975_ST2_REG      0x09
#define AK8975_CONTROL_REG  0x0A
#define AK8975_SELFTEST_REG 0x0C

/**
 * Bits
 */
#define AK8975_DRDY_BIT     0
#define AK8975_DERROR_BIT   2
#define AK8975_OFLOW_BIT    3

#define AK8975_SINGLE_MEASUREMENT 1

#define AK8975_SENSITIVITY  0.3

/** AK8975 magnetometer/digital compass simple library.
  *
  * Example:
  * @code
  * #include "mbed.h"
  * #include "AK8975.h"
  * 
  * DigitalOut led1(LED1);
  * Serial pc(USBTX, USBRX); // tx, rx
  * AK8975 mag(p9,p10,0x0E);
  * 
  * 
  * int main() {
  *     int data[3];
  *     if (mag.testConnection())
  *         pc.printf("Connection succeeded. \n");
  *     else
  *         pc.printf("Connection failed. \n");
  *     
  *      while(1) {
  *         led1=!led1;
  *         mag.startMeasurement();
  *         wait(0.5);                      //(or use the isReady() function)
  *         mag.getAll(data);
  *         pc.printf("X: %d", data[0]);
  *         pc.putc('\n');
  *         pc.printf("Y: %d", data[1]);
  *         pc.putc('\n');
  *         pc.printf("Z: %d", data[2]);
  *         pc.putc('\n');
  *     }
  * }
  * @endcode
  */
class AK8975 {
    public:
     /**
     * Constructor.
     *
     * @param sda - mbed pin to use for the SDA I2C line.
     * @param scl - mbed pin to use for the SCL I2C line.
     * @param I2Caddress - the I2C address of the device (0x0C - 0x0F)
     */
     AK8975(PinName sda, PinName scl, char address);
     
     /**
     * Checks connection with the device.
     *
     * @return - true if working connection, otherwise false
     */
     bool testConnection( void );
     
     /**
     * Checks if measurement is ready
     *
     * @return - true if available measurement, otherwise false
     */
     bool isReady( void );
     
     /**
     * Gets the X data
     *
     * @return - signed integer containing the raw data
     */
     int getX( void );
     
     /**
     * Gets the Y data
     *
     * @return - signed integer containing the raw data
     */
     int getY( void );
     
     /**
     * Gets the Z data
     *
     * @return - signed integer containing the raw data
     */
     int getZ( void );
     
     /**
     * Gets all the data, this is more efficient than calling the functions individually
     *
     * @param data - pointer to integer array with length 3 where data is stored (data[0]=X - data[1]=Y - data[2]=Z)
     */
     void getAll( int *data );
     
        
     
     /**
     * Checks if there is a data error due to reading at wrong moment
     *
     * @return - true for error, false for no error
     */
     bool getDataError( void );
     
     /**
     * Checks if a magnetic overflow happened (sensor saturation)
     *
     * @return - true for error, false for no error
     */
     bool getOverflow( void );
     
     /**
     * Starts a measurement cycle
     */
     void startMeasurement( void );
     
     
     
     private:
     I2C connection;
     char deviceAddress;
        
     /**
     * Writes data to the device
     *
     * @param adress - register address to write to
     * @param data - data to write
     */
     void write( char address, char data);
     
     /**
     * Read data from the device 
     *
     * @param adress - register address to write to
     * @return - data from the register specified by RA
     */
     char read( char adress);
     
     /**
     * Read multtiple regigsters from the device, more efficient than using multiple normal reads. 
     *
     * @param adress - register address to write to
     * @param length - number of bytes to read
     * @param data - pointer where the data needs to be written to 
     */
     void read( char adress, char *data, int length);
};      
#endif