#ifndef STATE_MANAGER_H
#define STATE_MANAGER_H

#include "State.h"
#include "MainMenu.h"
#include "Game.h"
#include "TitleScreen.h"
#include "GameOver.h"
#include "SubmitHighscore.h"

/** @file StateManager.h
* @author Andreas Garmannslund
* @date April 2015
*/

/// Finite State Machine for program flow
class StateManager
{
    public:
        /** Creates a new finite state machine. The states are defined in State.h
        * @param lcd Pointer to the lcd
        * @param input Pointer to the InputManager which is controlling user input.
        * @param firstState The initial state of the finite state machine.
        */
        StateManager(N5110 *lcd, InputManager* input, Sound *sound, MainState firstState)
                    : lcd(lcd),  input(input), sound(sound) {currentState = 0; nextState = NO_STATE; changeState(firstState);}
                    
        /// Frees allocated memory
        ~StateManager() {if (currentState != 0) delete currentState;}
        
        /// Update logic of the current state
        void update(float dt);
        
        /// Draw the current state to the lcd
        void render();
        
        /** Can be used to request the fsm to switch state.
        * @param newState The requested state
        */
        void requestStateChange(MainState newState);
        
        /// Sees if any requests to change the state have been made
        void processRequest();
        
    private:
        /** Deletes the current state and create a new one
        * @param newState The state which the finite state machine switches to
        */
        void changeState(MainState newState);
    
    // Variables
    private:
        N5110 *lcd;
        InputManager *input;
        Sound *sound;
        State* currentState; // Current state object
        MainState nextState; // requested state, NONE if no state is requested
};

#endif