/**
* @file main.cpp
* @brief Simple platform game developed for ELEC2645 Embedded Systems Project at University of Leeds
* @author Andreas Garmannslund
* @date April/May 2015
*/

#include "mbed.h"
#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"
#include "PinDetect.h"
#include <string>
#include <sstream>
#include <ctime>

#include "Joystick.h"
#include "StateManager.h"
#include "State.h"
#include "InputManager.h"
#include "Sound.h"

// Redefine pin names for simple access.
#define JOY_H p17
#define JOY_V p16
#define JOY_BTN p15

#define LED_POT p20

#define BUTTON_A p28
#define BUTTON_B p27
#define BUTTON_C p29

// Input and Output
/// Display
N5110 *lcd;

/// Sound: Piezo buzzer
Sound *sound; 

/// Responsible for managing user input
InputManager *input;

/// Brightness potentiometer
AnalogIn ledPot(LED_POT);

AnalogIn randomPin(p18); // Unconnected pin => noise, used for generating a random seed.
AnalogIn randomPin2(p19); // Unconnected pin

/// Set up initial variables
void init();

/// Frees remaining allocated memory
void cleanUp();

/// Finite state machine
StateManager* fsm;

/// Local file system
LocalFileSystem local("local");

/// Function for generating a random seed by using a unconnected analog input pin
// Problem: Not trully random as pin is fairly stable. It is unfortunately connected to the ground layer (often returning 0). Would be better to connect it to a "loose wire"/bad connection-
unsigned int getRandSeed();

/// Read and load the high score file.
void readHighscoreFile();

int main()
{    
    srand(getRandSeed());
    
    init();
    
    Timer timer;
    timer.start();
    
    while(true)
    {
        // We need to call Sleep() repeatedly, because periodic interrupts in InputManager will cause it to wake up every 20 ms.
        while (timer.read() < 0.08)
            Sleep();
        
        // update
        lcd->setBrightness(1.0 - ledPot); // Update brightness of screen
        fsm->update(timer.read());
        input->joystick->update();
        
        // render
        lcd->clear();
        fsm->render(); 
        lcd->refresh();
        
        fsm->processRequest(); // Change the state if requested.
        
        timer.reset();
    }
    
    cleanUp();  // Not really reached as the program never terminates. Added for completeness.
    
    return 0;
}

void init()
{
    // Disable ethernet to save power
    PHY_PowerDown();
    
    // Init LCD
    lcd = new N5110(p7, p8, p9, p10, p11, p13, p26);
    lcd->init();
    lcd->normalMode();
    lcd->setBrightness(1.0 - ledPot); // Update brightness of screen
    
    // Input
    input = new InputManager(BUTTON_A, BUTTON_B, BUTTON_C, JOY_H, JOY_V, JOY_BTN);
    
    // Sound
    sound = new Sound(p21);

    // Finite state machine
    fsm = new StateManager(lcd, input, sound, TITLE_SCREEN);
    
    readHighscoreFile(); // load highscores from file system
    
    sound->playNote(SFX::RESTART);
}

void readHighscoreFile()
{    
     FILE *fp = fopen("/local/highscores.txt", "r");    // read from local file
     
     if (!fp)       // if file is not created yet the highscores will default values as defined in Global.cpp
        return;
        
     char initials[4];
     char scoreStr[8];

     for (int i = 0; i < 3; ++i)
     {   
         fscanf(fp, "%s %s", initials, scoreStr);   // read from file
         
         int score = atoi(scoreStr);    // convert to int
         
         // update global highscore table
         Global::highscores[i].initials = initials;
         Global::highscores[i].score = score;
     }
        
     fclose(fp);
}

void cleanUp()
{
    delete lcd;
    delete input;
    delete sound;
}

/** Get a random seed based on unconnected analog input pins
  * @see https://developer.mbed.org/questions/2886/Why-is-the-rand-function-not-the-least-b/
 */
unsigned int getRandSeed()
{
    unsigned int randNum = 0;
    unsigned int lsb; // least significant bit
    
    for (unsigned int i = 0; i < 16; ++i)
    {
        lsb = 1 & (randomPin.read_u16()); // Read least significant bit in randomInput - value depends on noise
        randNum |= (lsb << i);              // Set bit nr.i to lsb in randomInput
        
        lsb = 1 & (randomPin2.read_u16());
        randNum |= (lsb << (16+i));
    }
    
    return randNum;
}