/**
@file Joystick.h
@brief Class used for controlling the button
@author Andreas Garmannslund
@date April 2015
*/

#ifndef JOYSTICK_H
#define JOYSTICK_H

#include "mbed.h"

/** @brief Enum used for the 8-directions of the joystick. */
enum JoystickDirection {CENTER, UP, DOWN, LEFT, RIGHT, UP_LEFT, UP_RIGHT, DOWN_LEFT, DOWN_RIGHT, UNKNOWN};

/// Joystick class
class Joystick
{   
    public:
    
        /** Creates a new Joystick object
        * @param x Pin connected to the horizontal potentiometer of the joystick
        * @param y Pin connected to the vertical potentiometer of the joystick
        * @param button Pin connected to the button of the thumb joystick
        */
        Joystick(PinName x, PinName y, PinName button);
        
        /** @brief Deconstructor. Frees allocated memory */
        ~Joystick();
        
        /** @brief Updates the current direction and button status of the joystick */
        void update();
        
        /** Calibrates the joystick. The joystick must be centered while this function is called */
        void calibrate(); // Calibrates joystick by updating the center positions
        
        /** Returns the current JoystickDirection based on last update
        * @return The current JoystickDirection.
        */
        int getDirection() {return dir;}
        
        /** Reads the value of the button
        * @return 1 if pressed, 0 otherwise
        */
        int readButton() {return *btn;};
        
        /** Square set around the center of the joystick where the input is ignored. The axes are treated seperately. Can be varied from 0 (no dead-zone) to 0.5 (max value for dx and dy). **/
        static const float DEAD_ZONE = 0.1; 
        
    private:
        AnalogIn *xPot;
        AnalogIn *yPot;
        DigitalIn *btn;
        float dx, dy;
        float centerX, centerY;
        JoystickDirection dir;
};

#endif