#include "mbed.h"

DigitalOut myled(LED1);

/*
    OLED        mbed
    p1 3.3V --- Vout
    p2 CS ----- p21
    p3 A0 ----- p23
    p4 SDA ---- p11
    p5 SCK ---- p13
    p6 RES ---- p22
    p7 GND ---- GND
*/

DigitalOut cs(p21);
DigitalOut rst(p22);
DigitalOut rs(p23);
SPI spi(p11, p12, p13); // MOSI MISO SCK

void send_cmd(uint8_t cmd){
    cs = rs = 0;
    spi.write(cmd);
    cs = 1;
}

void send_data(uint8_t data){
    cs = 0;
    rs = 1;
    spi.write(data);
    cs = 1;
}

void init(void)
{
    spi.format(8, 3);
    spi.frequency(1000000); // TBD
    cs = 1;
    rst = 0;
    wait_ms(5); // TBD
    rst = 1;
    wait_ms(1); // TBD
    send_cmd(0xae); // display off
    send_cmd(0xd5); // display divide ratio/osc. freq. ratio
    send_cmd(0x80);
    send_cmd(0xa8); // multiplex ation mode: 63
    send_cmd(0x3f);
    send_cmd(0xd3); // set display offset
    send_cmd(0x00);
    send_cmd(0x40); // set display start line
    send_cmd(0x8d); // set display offset
    send_cmd(0x14);
    send_cmd(0xa1); // segment remap
    send_cmd(0xc8); // set COM output scan direction
    send_cmd(0xda); // common pads hardware: alternative
    send_cmd(0x12);
    send_cmd(0x81); // contrast control
    send_cmd(0xcf);
    send_cmd(0xd9); // set pre-charge period
    send_cmd(0xf1);
    send_cmd(0xdb); // VCOM deselect level mode
    send_cmd(0x40); // set vcomh = 0.83 * VCC
    send_cmd(0xa4); // set entire display on/off
    send_cmd(0xa6); // set normal display
    send_cmd(0xaf); // set display on
}

// set position (x, 8*y)
void locate(int x, int y){
    send_cmd(0xb0 | (y & 0x0f)); // Page Address Set (see 2.4.3)
    send_cmd(0x10 | (x >> 4 & 0x0f)); // Column Address Set (see 2.4.4)
    send_cmd(x & 0x0f);
}

void cls(void){
    int x, y;
    for(y = 0; y < 8; y++){
        locate(0, y);
        for(x = 0; x < 128; x++) send_data(0x00);
    }
}

// miku1.bmp 75x64
const unsigned char miku1[75*8] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x64,0x08,0xf0,0x80,0x20,0xc8,0x90,0x28,0x40,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x60,0x50,0xa0,0x20,0x40,0xa0,0x00,0xe0,0x00,0x20,0xc0,0x00,0x80,0xc0,0xa0,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x06,0x08,0x17,0x2a,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xd8,0x3e,0xc5,0xba,0x21,0xab,0x04,0x00,0x03,0x3a,0x44,0xfb,0x00,0x07,0x08,0x75,0x83,0x56,0xad,0x1a,0xe4,0x89,0x22,0x8c,0x30,0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0xe0,0xe0,0xe0,0xe0,0xf0,0xcc,0xd1,0xbf,0xc0,0x3f,0xc0,0x03,0x00,0x82,0x10,0x10,0x10,0x00,0xf0,0x01,0x02,0x01,0x04,0x03,0x40,0xdd,0x32,0xf4,0x3f,0x3f,0x1c,0xe2,0x0c,0xf1,0x06,0xd8,0x20,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0xdf,0x3f,0x5f,0xbf,0xbf,0xbf,0xbf,0x3f,0xff,0xbf,0x3f,0x6f,0x0a,0xc2,0x74,0x94,0x74,0xc8,0xef,0x10,0x7c,0x94,0x0c,0x43,0x04,0xc3,0x01,0x00,0x00,0x00,0x00,0xdb,0x22,0x94,0x63,0x8c,0x31,0xc6,0x18,0xe0,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x4a,0x94,0x61,0x9e,0x61,0x9e,0x62,0x9c,0xe3,0xbc,0xcb,0x18,0xd6,0xa9,0xd6,0xa1,0xbe,0x49,0x92,0x0c,0xd1,0x2e,0xd1,0xff,0xff,0xff,0xff,0xfe,0xe0,0x80,0x00,0x00,0xff,0x20,0xca,0x15,0x4a,0xb0,0x45,0x12,0xed,0x12,0xe8,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0xbc,0x02,0xf5,0x00,0xff,0x00,0xff,0x00,0xff,0x04,0x5b,0xf7,0x1f,0x2f,0x5f,0x5f,0x7f,0xbe,0xbd,0xba,0x3c,0x7f,0x3c,0x1f,0xff,0xbf,0xff,0x3f,0xbf,0x3f,0x3f,0xee,0x2f,0xc8,0x32,0x84,0x29,0x54,0x81,0x7e,0x81,0x7c,0x83,0x7c,0xab,0x40,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x48,0xa3,0x5c,0xa3,0x4c,0xb3,0x4c,0x33,0x0c,0x0b,0x00,0x00,0x01,0xff,0xff,0xff,0xff,0xf4,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x3e,0xe0,0x0d,0xd0,0x25,0xda,0x25,0xda,0x25,0xda,0xa5,0x1a,0x01,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x01,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,0x1f,0x1f,0x0f,0x04,0x1f,0x1f,0x1f,0x1f,0x0f,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x3f,0x28,0x17,0x08,0x07,0x02,0x00,0x00,0x00,0x00,
};

// miku2.bmp 75x64
const unsigned char miku2[75*8] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x60,0x50,0xa0,0x20,0x40,0xa0,0x00,0xe0,0x00,0x20,0xc0,0x00,0x80,0xc0,0xa0,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x80,0x00,0x80,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xd8,0x3e,0xc5,0xba,0x21,0xab,0x04,0x00,0x03,0x3a,0x44,0xfb,0x00,0x07,0x08,0x75,0x83,0x56,0xad,0x1a,0xe4,0x89,0x22,0x8c,0x30,0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x34,0x41,0x15,0x2a,0x51,0x0e,0x30,0x0c,0x10,0x28,0x00,0x00,0x00,0x00,0x00,0x00,0x40,0x00,0x00,0x20,0x00,0x40,0xf0,0xf8,0xf8,0xfc,0xfc,0xfc,0xfc,0xf1,0xff,0xc0,0x3f,0xc0,0x03,0x00,0x82,0x10,0x10,0x10,0x00,0xf0,0x01,0x02,0x01,0x04,0x03,0x40,0xdd,0x32,0xf4,0x3f,0x3f,0x1c,0xe2,0x0c,0xf1,0x06,0xd8,0x20,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x83,0x33,0x47,0x97,0x6f,0x97,0x6f,0x8f,0x5f,0xbf,0x0f,0x27,0x02,0xc6,0x70,0x94,0x74,0xc8,0xef,0x10,0x7c,0x94,0x0c,0x43,0x04,0xc3,0x01,0x00,0x00,0x00,0x00,0xdb,0x22,0x94,0x63,0x8c,0x31,0xc6,0x18,0xe0,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x4a,0x95,0x60,0x9f,0x60,0x9f,0x60,0x9f,0xe0,0xbf,0xca,0x18,0xd6,0xa9,0xd6,0xa1,0xbe,0x49,0x92,0x0c,0xd1,0x2e,0xd1,0xff,0xff,0xff,0xff,0xfe,0xe0,0x80,0x00,0x00,0xff,0x20,0xca,0x15,0x4a,0xb0,0x45,0x12,0xed,0x12,0xe8,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0xbc,0x02,0xf5,0x00,0xff,0x00,0xff,0x00,0xff,0x04,0x5b,0xf7,0x1f,0x2f,0x5f,0x5f,0x7f,0xbe,0xbd,0xba,0x3c,0x7f,0x3c,0x1f,0xff,0xbf,0xff,0x3f,0xbf,0x3f,0x3f,0xee,0x2f,0xc8,0x32,0x84,0x29,0x54,0x81,0x7e,0x81,0x7c,0x83,0x7c,0xab,0x40,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x48,0xa3,0x5c,0xa3,0x4c,0xb3,0x4c,0x33,0x0c,0x0b,0x00,0x00,0x01,0xff,0xff,0xff,0xff,0xf4,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x3e,0xe0,0x0d,0xd0,0x25,0xda,0x25,0xda,0x25,0xda,0xa5,0x1a,0x01,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x01,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,0x1f,0x1f,0x0f,0x04,0x1f,0x1f,0x1f,0x1f,0x0f,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x3f,0x28,0x17,0x08,0x07,0x02,0x00,0x00,0x00,0x00,
};

// miku3.bmp 75x64
const unsigned char miku3[75*8] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x60,0x50,0xa0,0x20,0x40,0xa0,0x00,0xe0,0x00,0x20,0xc0,0x00,0x80,0xc0,0xa0,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xd8,0x3e,0xc5,0xba,0x21,0xab,0x04,0x00,0x03,0x3a,0x44,0xfb,0x00,0x07,0x08,0x75,0x83,0x56,0xad,0x1a,0xe4,0x89,0x22,0x8c,0x30,0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x30,0xcc,0x11,0x7f,0xc0,0x3f,0xc0,0x03,0x00,0x82,0x10,0x10,0x10,0x00,0xf0,0x01,0x02,0x01,0x04,0x03,0x40,0xdd,0x32,0xf4,0x3f,0x3f,0x1c,0xe2,0x0c,0xf1,0x06,0xd8,0x20,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0xc0,0xe8,0xc2,0xdd,0xe0,0xdf,0xe0,0xdf,0xe0,0xff,0xa1,0x03,0x82,0x42,0xf4,0x24,0x74,0xc8,0x2f,0xd0,0x3c,0x94,0x0c,0x43,0x04,0xc3,0x01,0x00,0x00,0x00,0x00,0xdb,0x22,0x94,0x63,0x8c,0x31,0xc6,0x18,0xe0,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x25,0x9f,0x7f,0x7f,0xff,0x7f,0xbf,0x3f,0xcf,0xff,0x37,0xcb,0x1d,0xd3,0xac,0xd3,0xac,0xb1,0x4f,0x90,0x0d,0xd0,0x2f,0xd0,0xff,0xff,0xff,0xff,0xfe,0xe0,0x80,0x00,0x00,0xff,0x20,0xca,0x15,0x4a,0xb0,0x45,0x12,0xed,0x12,0xe8,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xc0,0x00,0xc0,0x20,0xc0,0x20,0xa0,0x40,0x90,0x70,0x00,0x68,0x00,0x00,0x00,0x00,0x00,0x00,0x54,0xbb,0x44,0x89,0x32,0xcd,0x32,0xcd,0x32,0xcd,0x36,0x4b,0xf7,0x1f,0x2f,0x5f,0x5f,0x7f,0xbe,0xbd,0xba,0x3c,0x7f,0x3c,0x1f,0xff,0xbf,0xff,0x3f,0xbf,0x3f,0x3f,0xee,0x2f,0xc8,0x32,0x84,0x29,0x54,0x81,0x7e,0x81,0x7c,0x83,0x7c,0xab,0x40,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x00,0x02,0x01,0x04,0x08,0x03,0x04,0x02,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0e,0x51,0xa6,0x58,0xa3,0x4c,0xb3,0x4c,0x33,0x0c,0x0b,0x00,0x00,0x01,0xff,0xff,0xff,0xff,0xf4,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x3e,0xe0,0x0d,0xd0,0x25,0xda,0x25,0xda,0x25,0xda,0xa5,0x1a,0x01,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x01,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,0x1f,0x1f,0x0f,0x04,0x1f,0x1f,0x1f,0x1f,0x0f,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x3f,0x28,0x17,0x08,0x07,0x02,0x00,0x00,0x00,0x00,
};

// miku4.bmp 75x64
const unsigned char miku4[75*8] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x60,0x50,0xa0,0x20,0x40,0xa0,0x00,0xe0,0x00,0x20,0xc0,0x00,0x80,0xc0,0xa0,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xd8,0x3e,0xc5,0xba,0x21,0xab,0x04,0x00,0x03,0x3a,0x44,0xfb,0x00,0x07,0x08,0x75,0x83,0x56,0xad,0x1a,0xe4,0x89,0x22,0x8c,0x30,0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x30,0xcc,0x11,0x7f,0xc0,0x3f,0xc0,0x03,0x00,0x82,0x10,0x10,0x10,0x00,0xf0,0x01,0x02,0x01,0x04,0x03,0x40,0xdd,0x32,0xf4,0x3f,0x3f,0x1c,0xe2,0x0c,0xf1,0x06,0xd8,0x20,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0xc0,0xe0,0xf8,0xe2,0xfd,0xe0,0xff,0xe0,0xff,0xe0,0xff,0xe1,0x83,0x22,0xc2,0x64,0x94,0x74,0xc8,0xef,0x10,0x7c,0x94,0x0c,0x43,0x04,0xc3,0x01,0x00,0x00,0x00,0x00,0xdb,0x22,0x94,0x63,0x8c,0x31,0xc6,0x18,0xe0,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x80,0x40,0x00,0xc0,0x00,0xe0,0x00,0x40,0x00,0x00,0x00,0x02,0x00,0x00,0x02,0x00,0x00,0xa1,0x4f,0x1f,0xdf,0x3f,0xdf,0x3f,0xcf,0x2f,0xf7,0xcf,0x9b,0x2b,0xd3,0xac,0xd3,0xac,0xb1,0x4e,0x91,0x0c,0xd1,0x2e,0xd1,0xff,0xff,0xff,0xff,0xfe,0xe0,0x80,0x00,0x00,0xff,0x20,0xca,0x15,0x4a,0xb0,0x45,0x12,0xed,0x12,0xe8,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x23,0xd0,0x0a,0xd5,0x32,0x05,0x0c,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x54,0xb8,0x47,0x88,0x33,0xcc,0x33,0xc4,0x3d,0xc3,0x3c,0x47,0xf7,0x1f,0x2f,0x5f,0x5f,0x7f,0xbe,0xbd,0xba,0x3c,0x7f,0x3c,0x1f,0xff,0xbf,0xff,0x3f,0xbf,0x3f,0x3f,0xee,0x2f,0xc8,0x32,0x84,0x29,0x54,0x81,0x7e,0x81,0x7c,0x83,0x7c,0xab,0x40,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0e,0x51,0xa6,0x58,0xa3,0x4c,0xb3,0x4c,0x33,0x0c,0x0b,0x00,0x00,0x01,0xff,0xff,0xff,0xff,0xf4,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x3e,0xe0,0x0d,0xd0,0x25,0xda,0x25,0xda,0x25,0xda,0xa5,0x1a,0x01,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x01,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,0x1f,0x1f,0x0f,0x04,0x1f,0x1f,0x1f,0x1f,0x0f,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x3f,0x28,0x17,0x08,0x07,0x02,0x00,0x00,0x00,0x00,
};

// miku5.bmp 75x64
const unsigned char miku5[75*8] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x60,0x50,0xa0,0x20,0x40,0xa0,0x00,0xe0,0x00,0x20,0xc0,0x00,0x80,0xc0,0xa0,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xd8,0x3e,0xc5,0xba,0x21,0xab,0x04,0x00,0x03,0x3a,0x44,0xfb,0x00,0x07,0x08,0x75,0x83,0x56,0xad,0x1a,0xe4,0x89,0x22,0x8c,0x30,0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x30,0xcc,0x11,0x7f,0xc0,0x3f,0xc0,0x03,0x00,0x82,0x10,0x10,0x10,0x00,0xf0,0x01,0x02,0x01,0x04,0x03,0x40,0xdd,0x32,0xf4,0x3f,0x3f,0x1c,0xe2,0x0c,0xf1,0x06,0xd8,0x20,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x20,0x40,0x20,0x80,0x60,0x80,0x60,0x80,0x60,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x20,0x00,0x10,0x20,0x00,0x50,0xf0,0xf8,0xf8,0xfa,0xfd,0xf0,0xff,0xf0,0xff,0xf0,0xef,0xe1,0xa3,0x82,0x42,0xf4,0x24,0x74,0xc8,0x2f,0xd0,0x3c,0x94,0x0c,0x43,0x04,0xc3,0x01,0x00,0x00,0x00,0x00,0xdb,0x22,0x94,0x63,0x8c,0x31,0xc6,0x18,0xe0,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x0c,0x10,0x04,0x0a,0x15,0x08,0x07,0x00,0x02,0x01,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x21,0xcf,0x17,0x6f,0x9f,0x67,0x9f,0x67,0x9b,0xeb,0xb3,0xcf,0x19,0xd3,0xac,0xd3,0xac,0xb1,0x4f,0x90,0x0d,0xd0,0x2f,0xd0,0xff,0xff,0xff,0xff,0xfe,0xe0,0x80,0x00,0x00,0xff,0x20,0xca,0x15,0x4a,0xb0,0x45,0x12,0xed,0x12,0xe8,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x54,0xb9,0x46,0x11,0xcc,0x33,0xcc,0x33,0xcc,0x33,0x4c,0x57,0xf7,0x1f,0x2f,0x5f,0x5f,0x7f,0xbe,0xbd,0xba,0x3c,0x7f,0x3c,0x1f,0xff,0xbf,0xff,0x3f,0xbf,0x3f,0x3f,0xee,0x2f,0xc8,0x32,0x84,0x29,0x54,0x81,0x7e,0x81,0x7c,0x83,0x7c,0xab,0x40,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0e,0x51,0xa6,0x58,0xa7,0x50,0xaf,0x50,0x2d,0x13,0x0c,0x03,0x00,0x00,0xff,0xff,0xff,0xff,0xf4,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x3e,0xe0,0x0d,0xd0,0x25,0xda,0x25,0xda,0x25,0xda,0xa5,0x1a,0x01,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x01,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,0x1f,0x1f,0x0f,0x04,0x1f,0x1f,0x1f,0x1f,0x0f,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x3f,0x28,0x17,0x08,0x07,0x02,0x00,0x00,0x00,0x00,
};

// miku6.bmp 75x64
const unsigned char miku6[75*8] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x60,0x50,0xa0,0x20,0x40,0xa0,0x00,0xe0,0x00,0x20,0xc0,0x00,0x80,0xc0,0xa0,0x40,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0a,0x00,0x3e,0x00,0x34,0x28,0x43,0x3c,0x41,0xac,0x50,0xa0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0xd8,0x3e,0xc5,0xba,0x21,0xab,0x04,0x00,0x03,0x3a,0x44,0xfb,0x00,0x07,0x08,0x75,0x83,0x56,0xad,0x1a,0xe4,0x89,0x22,0x8c,0x30,0xc0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x40,0x80,0xc0,0xc0,0xc0,0xf0,0x8c,0xd1,0x3f,0xc0,0x3f,0xc0,0x03,0x00,0x82,0x10,0x10,0x10,0x00,0xf0,0x01,0x02,0x01,0x04,0x03,0x40,0xdd,0x32,0xf4,0x3f,0x3f,0x1c,0xe2,0x0c,0xf1,0x06,0xd8,0x20,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x1f,0x3f,0xff,0x3f,0xff,0x3f,0xff,0x3f,0x7f,0x7f,0xff,0x3d,0x6b,0x0a,0xc2,0x74,0x94,0x74,0xc8,0xef,0x10,0x7c,0x94,0x0c,0x43,0x04,0xc3,0x01,0x00,0x00,0x00,0x00,0xdb,0x22,0x94,0x63,0x8c,0x31,0xc6,0x18,0xe0,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xa0,0x8a,0x35,0xc0,0x3f,0xc0,0x3f,0x40,0xdd,0xf3,0x34,0xcf,0x18,0xf2,0x8d,0xea,0x91,0x7e,0x09,0xd2,0x0c,0xd1,0x2e,0xd1,0xff,0xff,0xff,0xff,0xfe,0xe0,0x80,0x00,0x00,0xff,0x20,0xca,0x15,0x4a,0xb0,0x45,0x12,0xed,0x12,0xe8,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x52,0xbc,0x02,0xf5,0x00,0xff,0x00,0xff,0x00,0xff,0x04,0x5b,0xf7,0x1f,0x2f,0x5f,0x5f,0x7f,0xbf,0xbe,0xb8,0x3d,0x7e,0x3c,0x1f,0xff,0xbf,0xff,0x3f,0xbf,0x3f,0x3f,0xee,0x2f,0xc8,0x32,0x84,0x29,0x54,0x81,0x7e,0x81,0x7c,0x83,0x7c,0xab,0x40,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x48,0xa3,0x5c,0xa3,0x4c,0xb3,0x4c,0x33,0x0c,0x0b,0x00,0x00,0x01,0xff,0xff,0xff,0xff,0xf4,0x00,0xff,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x3e,0xe0,0x0d,0xd0,0x25,0xda,0x25,0xda,0x25,0xda,0xa5,0x1a,0x01,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x01,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x0f,0x1f,0x1f,0x0f,0x04,0x1f,0x1f,0x1f,0x1f,0x0f,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x3f,0x28,0x17,0x08,0x07,0x02,0x00,0x00,0x00,0x00,
};

const unsigned char *miku[6] = { miku1, miku2, miku3, miku4, miku5, miku6 };

void drawmiku(int pic){
    int x, y;
    const unsigned char *p = miku[pic];

    for(y = 0; y < 8; y++){
        locate(53, y);
        for(x = 0; x < 75; x++) send_data(*p++);
    }
}

int main() {
    int pic;

    init();
    
    cls();
    pic = 0;
    while(1) {
        drawmiku(pic);
        if(++pic == 6) pic = 0;

        myled = !myled;
        wait_ms(100);
    }
}
