/*
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mbed.h"
#include "rga_func.h"
#include "DisplayBace.h"
#include "rtos.h"


/* LCD Parameter */
#define LCD_INPUT_CLOCK                     (66.67f)
#define LCD_OUTPUT_CLOCK                    (33.26f)

#define LCD_PIXEL_WIDTH                     (800)
#define LCD_PIXEL_HEIGHT                    (480)
#define LCD_H_BACK_PORCH                    (128 + 36)
#define LCD_H_FRONT_PORCH                   (92)
#define LCD_V_BACK_PORCH                    (35 + 5)
#define LCD_V_FRONT_PORCH                   (5)

/* FRAME BUFFER Parameter */
#define FRAME_BUFFER_BYTE_PER_PIXEL         (2)
#define FRAME_BUFFER_STRIDE                 (((LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL) + 31u) & ~31u)

#define DRAW_RECTANGLE_CNT_MAX              (4)

typedef enum {
    RGA_FUNC_NON,
    RGA_FUNC_DRAW_RECTANGLE,
    RGA_FUNC_DRAW_IMAGE,
    RGA_FUNC_DISSOLVE,
    RGA_FUNC_SCROLL,
    RGA_FUNC_ZOOM,
    RGA_FUNC_ROTATION,
    RGA_FUNC_ACCELERATE,
    RGA_FUNC_ANIME_EASE,
    RGA_FUNC_ANIME_LINEAR,
    RGA_FUNC_ANIME_EASE_IN,
    RGA_FUNC_ANIME_EASE_OUT,
    RGA_FUNC_ANIME_EASE_IN_OUT,
    RGA_FUNC_RETURN,
    RGA_FUNC_END
}func_code_t;

DigitalOut  lcd_pwon(P7_15);
DigitalOut  lcd_blon(P8_1);
PwmOut      lcd_cntrst(P8_15);
I2C         i2c(I2C_SDA, I2C_SCL);
DisplayBase Display;

typedef struct {
    uint8_t y_h: 3,
    reserved: 1,
    x_h: 3,
    valid: 1;
    uint8_t x_l;
    uint8_t y_l;
    uint8_t z;
} xyz_data_t;
 
typedef struct {
    uint8_t fingers: 4,
    reserved: 4;
    uint8_t keys;
    xyz_data_t xyz_data;
} stx_report_data_t;

typedef struct {
    uint32_t pic_pos_x;             /* X position of the key picture. */
    uint32_t pic_pos_y;             /* Y position of the key picture. */
    uint32_t pic_width;             /* Width of the key picture. */
    uint32_t pic_height;            /* Height of the key picture. */
    func_code_t func_code;          /* func code of the key picture. */
} key_pic_info_t;

static uint8_t user_frame_buffer[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT]__attribute((aligned(32)));  /* 32 bytes aligned */
static uint8_t user_frame_buffer2[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT]__attribute((aligned(32))); /* 32 bytes aligned */
static frame_buffer_t frame_buffer_info;
static volatile int32_t vsync_count = 0;

static const key_pic_info_t top_screen_key_tbl[] = {
    /*   X       Y     Width   Height   Func code                 */
    {    50,    350,    120,     52,    RGA_FUNC_DRAW_RECTANGLE    },  /* RGA Func1 */
    {   230,    350,    120,     52,    RGA_FUNC_DRAW_IMAGE        },  /* RGA Func2 */
    {   410,    350,    120,     52,    RGA_FUNC_DISSOLVE          },  /* RGA Func3 */
    {    50,    420,    120,     52,    RGA_FUNC_SCROLL            },  /* RGA Func4 */
    {   230,    420,    120,     52,    RGA_FUNC_ZOOM              },  /* RGA Func5 */
    {   410,    420,    120,     52,    RGA_FUNC_ROTATION          },  /* RGA Func6 */
    {   615,    420,    120,     52,    RGA_FUNC_ACCELERATE        },  /* RGA Func7 */
    {     0,      0,      0,      0,    RGA_FUNC_END               }   /* table end */
};

static const key_pic_info_t return_key_tbl[] = {
    /*   X       Y     Width   Height   Func code                 */
    {   640,     10,    150,     84,    RGA_FUNC_RETURN            },  /* Return Top Screen */
    {     0,      0,      0,      0,    RGA_FUNC_END               }   /* table end */
};

static const key_pic_info_t animetion_timing_key_tbl[] = {
    /*   X       Y     Width   Height   Func code                 */
    {   640,     10,    150,     84,    RGA_FUNC_RETURN            },  /* Return Top Screen */
    {    17,    372,    136,     50,    RGA_FUNC_ANIME_EASE        },  /* ease */
    {   173,    372,    136,     50,    RGA_FUNC_ANIME_LINEAR      },  /* linear */
    {   330,    372,    136,     50,    RGA_FUNC_ANIME_EASE_IN     },  /* ease-in */
    {   487,    372,    136,     50,    RGA_FUNC_ANIME_EASE_OUT    },  /* ease-out */
    {   644,    372,    136,     50,    RGA_FUNC_ANIME_EASE_IN_OUT },  /* ease-in-out */
    {     0,      0,      0,      0,    RGA_FUNC_END               }   /* table end */
};

static void IntCallbackFunc_Vsync(DisplayBase::int_type_t int_type) {
    /* Interrupt callback function for Vsync interruption */
    if (vsync_count > 0) {
        vsync_count--;
    }
}

static void Wait_Vsync(const int32_t wait_count) {
    /* Wait for the specified number of times Vsync occurs */
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

static void Init_LCD_Display(uint8_t* disp_buf) {
    /* Create DisplayBase object */
    DisplayBase::graphics_error_t error;
    DisplayBase::rect_t rect;
    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };

    lcd_pwon = 0;
    lcd_blon = 0;
    Thread::wait(100);
    lcd_pwon = 1;
    lcd_blon = 1;
    Thread::wait(100);

    lcd_config.lcd_type             = DisplayBase::LCD_TYPE_LVDS;
    lcd_config.intputClock          = LCD_INPUT_CLOCK;
    lcd_config.outputClock          = LCD_OUTPUT_CLOCK;
    lcd_config.lcd_outformat        = DisplayBase::LCD_OUTFORMAT_RGB888;
    lcd_config.lcd_edge             = DisplayBase::EDGE_RISING;
    lcd_config.h_toatal_period      = (LCD_PIXEL_WIDTH  + LCD_H_FRONT_PORCH + LCD_H_BACK_PORCH);
    lcd_config.v_toatal_period      = (LCD_PIXEL_HEIGHT + LCD_V_FRONT_PORCH + LCD_V_BACK_PORCH);

    lcd_config.h_disp_widht         = LCD_PIXEL_WIDTH;
    lcd_config.v_disp_widht         = LCD_PIXEL_HEIGHT;
    lcd_config.h_back_porch         = LCD_H_BACK_PORCH;
    lcd_config.v_back_porch         = LCD_V_BACK_PORCH;

    lcd_config.h_sync_port          = DisplayBase::LCD_TCON_PIN_NON;
    lcd_config.h_sync_port_polarity = DisplayBase::SIG_POL_NOT_INVERTED;
    lcd_config.h_sync_width         = 0;

    lcd_config.v_sync_port          = DisplayBase::LCD_TCON_PIN_NON;
    lcd_config.v_sync_port_polarity = DisplayBase::SIG_POL_NOT_INVERTED;
    lcd_config.v_sync_width         = 0;

    lcd_config.de_port              = DisplayBase::LCD_TCON_PIN_3;
    lcd_config.de_port_polarity     = DisplayBase::SIG_POL_NOT_INVERTED;

    /* Graphics initialization process */
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, IntCallbackFunc_Vsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);
    rect.vs = 0;
    rect.vw = LCD_PIXEL_HEIGHT;
    rect.hs = 0;
    rect.hw = LCD_PIXEL_WIDTH;

    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_0,
        (void *)disp_buf,
        FRAME_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_RGB565,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        &rect
    );
}

static void Start_LCD_Display(void) {
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);

    lcd_cntrst.write(1.0);
}

static void Update_LCD_Display(frame_buffer_t * frmbuf_info) {
    Display.Graphics_Read_Change(DisplayBase::GRAPHICS_LAYER_0,
     (void *)frmbuf_info->buffer_address[frmbuf_info->draw_buffer_index]);
    Wait_Vsync(1);
}

static int Get_Coordinates(uint8_t *count, uint32_t *x0, uint32_t *y0) {
    char buf[6];
    stx_report_data_t *pdata;
    int ret = -1;
    *count = 0;                     // Set point detected count to 0

    if (i2c.read((0x55 << 1), buf, sizeof(buf)) == 0) {
        pdata = (stx_report_data_t *)buf;
        if (pdata->fingers) {
            if (pdata->xyz_data.valid) {
                *x0 = (pdata->xyz_data.x_h << 8) | pdata->xyz_data.x_l;
                *y0 = (pdata->xyz_data.y_h << 8) | pdata->xyz_data.y_l;
                (*count)++;
            }
        }
        ret = 0;
    }

    return ret;
}

static func_code_t Scan_Key(const key_pic_info_t * key_tbl, const uint32_t pos_x, const uint32_t pos_y) {
    func_code_t ret = RGA_FUNC_NON;

    while (ret == RGA_FUNC_NON) {
        if (key_tbl->func_code == RGA_FUNC_END) {
            break;
        }
        /* Check the range of the X position */
        if ((pos_x >= key_tbl->pic_pos_x) && (pos_x <= (key_tbl->pic_pos_x + key_tbl->pic_width))) {
            /* Check the range of the Y position */
            if ((pos_y >= key_tbl->pic_pos_y) && (pos_y <= (key_tbl->pic_pos_y + key_tbl->pic_height))) {
                /* Decide the func code. */
                ret = key_tbl->func_code;
            }
        }
        key_tbl++;
    }

    return ret;
}

static void Swap_FrameBuffer(frame_buffer_t * frmbuf_info) {
    if (frmbuf_info->draw_buffer_index == 1) {
        frmbuf_info->draw_buffer_index = 0;
    } else {
        frmbuf_info->draw_buffer_index = 1;
    }
}

static void Exe_RGA_Func(int func_name, frame_buffer_t* frmbuf_info) {
    uint8_t touch_num = 0;
    uint32_t pos_x0 = 0;
    uint32_t pos_y0 = 0;
    func_code_t func_code;

    switch (func_name) {
        case RGA_FUNC_DRAW_RECTANGLE:
            bool key_on = false;
            int cnt;
            int color_cnt = 0;
            int x_0 = 0;
            int y_0 = 0;
            draw_rectangle_pos_t pos_tbl[DRAW_RECTANGLE_CNT_MAX] = {0};

            pos_tbl[0].style = "#FF0000";  /* red */
            pos_tbl[1].style = "#00FF00";  /* green */
            pos_tbl[2].style = "#0000FF";  /* blue */
            pos_tbl[3].style = "#000000";  /* black */

            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    if (Scan_Key(return_key_tbl, pos_x0, pos_y0) == RGA_FUNC_RETURN) {
                        break;
                    }
                    if (key_on == false) {
                        key_on = true;
                        if (color_cnt == 0) {
                            for (cnt = 0; cnt < DRAW_RECTANGLE_CNT_MAX; cnt++) {
                                pos_tbl[cnt].x = 0;
                                pos_tbl[cnt].y = 0;
                                pos_tbl[cnt].w = 0;
                                pos_tbl[cnt].h = 0;
                            }
                        }
                        x_0 = pos_x0;
                        y_0 = pos_y0;
                    }
                    if (x_0 < pos_x0) {
                        pos_tbl[color_cnt].x = x_0;
                        pos_tbl[color_cnt].w = pos_x0 - x_0;
                    } else {
                        pos_tbl[color_cnt].x = pos_x0;
                        pos_tbl[color_cnt].w = x_0 - pos_x0;
                    }
                    if (y_0 < pos_y0) {
                        pos_tbl[color_cnt].y = y_0;
                        pos_tbl[color_cnt].h = pos_y0 - y_0;
                    } else {
                        pos_tbl[color_cnt].y = pos_y0;
                        pos_tbl[color_cnt].h = y_0 - pos_y0;
                    }
                } else {
                    if (key_on != false) {
                        color_cnt++;
                        if (color_cnt == DRAW_RECTANGLE_CNT_MAX) {
                            color_cnt = 0;
                        }
                    }
                    key_on = false;
                }
                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                RGA_Func_DrawRectangle(frmbuf_info, pos_tbl, DRAW_RECTANGLE_CNT_MAX);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        case RGA_FUNC_DRAW_IMAGE:
            int center_pos_x = 320;
            int center_pos_y = 110;
            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    if (Scan_Key(return_key_tbl, pos_x0, pos_y0) == RGA_FUNC_RETURN) {
                        break;
                    }
                    center_pos_x = pos_x0;
                    center_pos_y = pos_y0;
                }
                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                RGA_Func_DrawImage(frmbuf_info, center_pos_x, center_pos_y);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        case RGA_FUNC_DISSOLVE:
            float32_t work_alpha = 0.0f;
            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    if (Scan_Key(return_key_tbl, pos_x0, pos_y0) == RGA_FUNC_RETURN) {
                        break;
                    }
                    work_alpha = (float32_t)pos_x0 / (float32_t)(LCD_PIXEL_WIDTH);
                }
                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                RGA_Func_Dissolve(frmbuf_info, work_alpha);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        case RGA_FUNC_SCROLL:
            int work_width_pos = 0;
            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    if (Scan_Key(return_key_tbl, pos_x0, pos_y0) == RGA_FUNC_RETURN) {
                        break;
                    }
                    work_width_pos = SCROLL_MAX_NUM * ((float32_t)pos_x0 / (float32_t)(LCD_PIXEL_WIDTH));
                }
                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                RGA_Func_Scroll(frmbuf_info, work_width_pos);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        case RGA_FUNC_ZOOM:
            int work_height_pos = ZOOM_MAX_NUM;
            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    if (Scan_Key(return_key_tbl, pos_x0, pos_y0) == RGA_FUNC_RETURN) {
                        break;
                    }
                    work_height_pos = ZOOM_MAX_NUM * ((float32_t)pos_x0 / (float32_t)(LCD_PIXEL_WIDTH));
                }
                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                RGA_Func_Zoom(frmbuf_info,  work_height_pos);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        case RGA_FUNC_ROTATION:
            graphics_matrix_float_t work_angle = 0;
            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    if (Scan_Key(return_key_tbl, pos_x0, pos_y0) == RGA_FUNC_RETURN) {
                        break;
                    }
                    work_angle = ROTATION_MAX_NUM * ((float32_t)pos_x0 / (float32_t)(LCD_PIXEL_WIDTH));
                }
                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                RGA_Func_Rotation(frmbuf_info, work_angle);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        case RGA_FUNC_ACCELERATE:
            int acce_frame_num = 0;
            int animation_timing = 0;
            float32_t work_relative_pos;
            while (1) {
                /* Get coordinates */
                Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                if (touch_num != 0) {
                    func_code_t func_code;

                    func_code = Scan_Key(animetion_timing_key_tbl, pos_x0, pos_y0);
                    if (func_code == RGA_FUNC_RETURN) {
                        break;
                    }
                    switch (func_code) {
                        case RGA_FUNC_ANIME_EASE:
                            animation_timing = 0;
                            acce_frame_num = 0;
                            break;
                        case RGA_FUNC_ANIME_LINEAR:
                            animation_timing = 1;
                            acce_frame_num = 0;
                            break;
                        case RGA_FUNC_ANIME_EASE_IN:
                            animation_timing = 2;
                            acce_frame_num = 0;
                            break;
                        case RGA_FUNC_ANIME_EASE_OUT:
                            animation_timing = 3;
                            acce_frame_num = 0;
                            break;
                        case RGA_FUNC_ANIME_EASE_IN_OUT:
                            animation_timing = 4;
                            acce_frame_num = 0;
                            break;
                        default:
                            /* Do Nothing */
                            break;
                    }
                }
                work_relative_pos = acce_frame_num / (float32_t)ACCELERATE_MAX_NUM;
                acce_frame_num++;
                if (acce_frame_num > ACCELERATE_MAX_NUM) {
                    acce_frame_num = 0;
                }

                /* Draw screen */
                Swap_FrameBuffer(frmbuf_info);
                //RGA_Func_Training(frmbuf_info, animation_timing, work_relative_pos);
                RGA_Func_Accelerate(frmbuf_info, animation_timing, work_relative_pos);
                Update_LCD_Display(frmbuf_info);
            }
            break;
        default :
            /* Do nothing */
            break;
    }
}

int main(void) {
    func_code_t func_code;
    uint8_t touch_num = 0;
    uint32_t pos_x0 = 0;
    uint32_t pos_y0 = 0;

    memset(user_frame_buffer, 0, sizeof(user_frame_buffer));
    memset(user_frame_buffer2, 0, sizeof(user_frame_buffer2));
    frame_buffer_info.buffer_address[0] = user_frame_buffer;
    frame_buffer_info.buffer_address[1] = user_frame_buffer2;
    frame_buffer_info.buffer_count      = 2;
    frame_buffer_info.show_buffer_index = 0;
    frame_buffer_info.draw_buffer_index = 0;
    frame_buffer_info.width             = LCD_PIXEL_WIDTH;
    frame_buffer_info.byte_per_pixel    = FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.stride            = LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.height            = LCD_PIXEL_HEIGHT;
    frame_buffer_info.pixel_format      = PIXEL_FORMAT_RGB565;
    Init_LCD_Display(frame_buffer_info.buffer_address[0]);

    /* Display Top Screen */
    Set_RGAObject(&frame_buffer_info);
    RGA_Func_DrawTopScreen(&frame_buffer_info);
    Start_LCD_Display();

    while (1) {
        /* Get Coordinates */
        Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
        if (touch_num != 0) {
            func_code = Scan_Key(top_screen_key_tbl, pos_x0, pos_y0);
            if (func_code != RGA_FUNC_NON) {
                /* Wait key off */
                while (1) {
                    Get_Coordinates(&touch_num, &pos_x0, &pos_y0);
                    if (touch_num == 0) {
                        break;
                    }
                    Thread::wait(50);
                }

                /* Execute RGA functions */
                Exe_RGA_Func(func_code, &frame_buffer_info);

                /* Return Top Screen */
                Swap_FrameBuffer(&frame_buffer_info);
                RGA_Func_DrawTopScreen(&frame_buffer_info);
                Update_LCD_Display(&frame_buffer_info);
            }
        }
        Thread::wait(100);
    }
}

#if 0
errnum_t  Sample_Training( void )
{
    enum { layer_num = 0 };
    enum { frame_num_max = 30 };

    errnum_t                        e;                      /* error code                   */
    int                             frame_num;              /* frame number for transition  */
    S_AcceleratedMotionWorkClass*   work = &g_S_Training;   /* struct for window surface    */
    graphics_t*                     graphics;               /* struct for graphics context  */
    frame_buffer_t*                 frame;                  /* information for frame buffer */
    float32_t                       position;               /* x-coordinate for drawing     */


    /* 変数の初期化 (Section: 6.9.9.2, 6.9.1.2) */
    R_WINDOW_SURFACES_InitConst( &work->WindowSurfaces );
    R_GRAPHICS_InitConst( &work->Graphics );

    /* Set "work->WindowSurfaces" */
    {
        window_surfaces_config_t  config;

        config.flags = F_WINDOW_SURFACES_PIXEL_FORMAT;
        config.pixel_format = PIXEL_FORMAT_RGB565;

        /* ウィンドウサーフェスの初期化 (Section: 6.9.9.3) */
        e= R_WINDOW_SURFACES_Initialize( &work->WindowSurfaces, &config );
        IF(e)
        {
            goto fin;
        }
    }

    /* Set "work->FrameBuffer" */
    /* 指定したレイヤのフレームバッファのポインタを取得 (Section: 6.9.9.5) */
    e= R_WINDOW_SURFACES_GetLayerFrameBuffer( &work->WindowSurfaces, layer_num, &work->FrameBuffer );
    IF(e)
    {
        goto fin;
    }

    /* Set "work->Graphics" */
    {
        graphics_config_t  config;

        config.flags = F_GRAPHICS_FRAME_BUFFER;
        config.frame_buffer = work->FrameBuffer;

        /* コンテキストの初期化 (Section: 6.9.1.3) */
        e= R_GRAPHICS_Initialize( &work->Graphics, &config );
        IF(e)
        {
            goto fin;
        }
    }


    /* アニメーションのタイミングを取得 (Section: 6.9.13.2) */
    frame = work->FrameBuffer;
    graphics = &work->Graphics;
    e= R_Get_AnimationTimingFunction( "ease-out", &work->Accelerator );
    IF(e)
    {
        goto fin;
    }

Loop:

    /* Do accelerated motion */
    /* 描画処理ループ */
    /* 一回のループ処理で一回分のアニメーション処理を実現 */
    for ( frame_num = 0;  frame_num <= frame_num_max;  frame_num += GS_FRAME_STEP )
    {
        const float32_t  angle = (float32_t)(( frame_num * 30) % 360);

        /* フレームバッファのクリア (Section: 6.9.1.28) */
        e= R_GRAPHICS_Clear( graphics, 0, 0, frame->width, frame->height );
        IF(e) { goto fin; }
        /*---------------------------------------------------------------------*/

        /*---------------------------------------------------------------------*/
        /* 画像の描画 (Section: 6.9.1.29) */
        e= R_GRAPHICS_DrawImage( graphics, g_JPEG_jpg_File, 0, 0);
        IF(e) { goto fin; }

        /* 表示バッファと描画バッファを入れ替え (Section: 6.9.9.7) */
        e= R_WINDOW_SURFACES_SwapBuffers( &work->WindowSurfaces, layer_num, graphics );
        IF(e) { goto fin; }

        if ( frame_num == 0 )
        {
            e= R_OSPL_Delay( 1000 );
            IF(e) { goto fin; }
        }
    }


    /* Stay */
    /* 描画終了待ち (Section: 6.9.9.10) */
    e= R_WINDOW_SURFACES_DoMessageLoop( &work->WindowSurfaces );
    IF(e) { goto fin; }
    e=0;

    goto Loop;
    
fin:
    /* 終了処理 */

    /* コンテキストの破棄 (Section: 6.9.1.4 and 6.9.9.4) */
    e= R_GRAPHICS_Finalize( &work->Graphics, e );
    e= R_WINDOW_SURFACES_Finalize( &work->WindowSurfaces, e );
    R_DEBUG_BREAK_IF_ERROR();
    return  e;
}

#ifdef NEVER_COMES_HERE
        /* animation */
        /* アニメーション処理時のX座標を取得 (Section: 6.9.13.3) */
        position = R_ANIMATION_TIMING_FUNCTION_GetValue( work->Accelerator,
                   (float32_t) frame_num / frame_num_max,  INITIAL_POSITION_Xf, 100.0f);

        /* 画像の描画 (Section: 6.9.1.29) */
        e= R_GRAPHICS_DrawImage( graphics, g_JPEG_jpg_File, (int_fast32_t) position, INITIAL_POSITION_Yi);              IF(e) { goto fin; }
        /* animation */


        /* rotate */
        /* 原点中心で回転処理 */
        e= R_GRAPHICS_RotateMatrixDegree( graphics, angle );                                                            IF(e) { goto fin;   }

        /* 原点が左上の画像に対して、画像の中心が原点となるように平行移動 */
        e= R_GRAPHICS_TranslateMatrix( graphics,
                                (graphics_matrix_float_t)(-JPEG_WIDTH / 2), 
                                (graphics_matrix_float_t)(-JPEG_HEIGHT / 2) );                                          IF(e) { goto fin;   }

        /* 希望する位置への移動 */
        e= R_GRAPHICS_TranslateMatrix(graphics, position, 100.0f);                                                      IF(e) { goto fin;   }

        /* 挿入画像の描画 (Section: 6.9.1.29) */
        e= R_GRAPHICS_DrawImage( graphics, g_JPEG_jpg_File, 0, 0 );                             IF(e) { goto fin; }
        /* rotate */

        /* clipping */
        /* クリッピングされた画像の描画 (Section: 6.9.1.29) */ 
        e= R_GRAPHICS_DrawImageChild( graphics, g_sample00_jpg_file, 
                                        0, 0, WVGA_WIDTH - (int_fast32_t)position, WVGA_HEIGHT, 
                                        (int_fast32_t) position, 0, WVGA_WIDTH - (int_fast32_t)position, WVGA_HEIGHT);  IF(e) { goto fin; }
        /* clipping */

        /* alpha blending */
        /* アルファ値の設定 */
        alpha = (uint8_t) (frame_num * 8) + 15;
        e= R_GRAPHICS_SetGlobalAlpha(graphics, alpha);                                          IF(e) { goto fin; }
        /* alpha blending */

#endif
#endif