/**
 * Current Log-O-Matic
 * Author: Shawn Hymel @ SparkFun Electronics
 * Date: July 1, 2015
 *
 * Measures current using an INA169 breakout board. Stores current and timestamp
 * to an SD card.
 *
 * INA169 Vout -> Pin 15
 * SD Card DI -> Pin 5
 * SD Card DO -> Pin 6
 * SD Card SCK -> Pin 7
 * SD Card CS -> Pin 8
 *
 * This code is beerware; if you see me (or any other SparkFun 
 * employee) at the local, and you've found our code helpful, please
 * buy us a round!
 *
 * Distributed as-is; no warranty is given.
 */
 
#include "mbed.h"
#include "SDFileSystem.h"

#define DEBUG       1       // 1 to print to console. 0 to turn off.
#define MAX_FILES   100     // Maximum number of log files to store on SD
#define DELAY_MS    0.02    // Time between samples (in seconds)
#define NUM_SAMPLES 500     // Number of samples to collect
#define RS          0.5     // Value of RS (sensing resistor) in ohms
#define RL          10000   // Value of RL (output resistor) in ohms

// Digital output (status LED)
DigitalOut status_led(LED1);

// Analog input (pin 15)
AnalogIn ain(p15);

// USB serial (tx, rx)
Serial pc(USBTX, USBRX);

// SD card (SPI pins)
SDFileSystem sd(p5, p6, p7, p8, "sd");

// Global variables
const char file_root[] = "idata";
const char file_suffix[] = ".csv";
char filename[20];
Timer timer;

// Return true if file (or directory) 'filename' is present in directory 'root'
// Based on: https://developer.mbed.org/forum/mbed/topic/463/
bool exists(char * root, char *filename)
{
    DIR *d = opendir(root);
    struct dirent *p;
    bool found = false;
    if ( d != NULL )
    {
        while ( !found && (p = readdir(d)) != NULL )
        {
            if ( strcmp(p->d_name, filename) == 0 )
                found = true;
        }
    }
    closedir(d);
    return found;
}

// Main
int main( void )
{
    FILE *file;
    char str[5];
    int i;
    int led_state;
    float voltage_in;
    float current;
    float timestamp;
    
    // Debugging start
#if DEBUG == 1
    printf("\n\r");
    printf("-------------------\n\r");
    printf("Current Log-O-Matic\n\r");
    printf("-------------------\n\r");
#endif
    
    // Create next file in sequence
    for ( i = 0; i < MAX_FILES; i++ ) {
           
        // Construct filename
        strcpy(filename, file_root);
        sprintf(str, "%d", i);
        strcat(filename, str);
        strcat(filename, file_suffix);
    
        // If file exists, skip to next. If not, create it.
        if ( exists("/sd", filename) ) {
#if DEBUG == 1
            printf ("%s found\n\r", filename);
#endif
            if ( i == (MAX_FILES - 1) ) {
                error("ERROR: Too many files!\n\r");
                return -1;
            }
        } else {
#if DEBUG == 1
            printf ("%s not found. Creating.\n\r", filename);
#endif
            break;
        }
    }
        
    // Construct absolute path to file
    strcpy(filename, "/sd/");
    strcat(filename, file_root);
    sprintf(str, "%d", i);
    strcat(filename, str);
    strcat(filename, file_suffix);
    
    // Open file for writing
#if DEBUG == 1
    printf("Opening %s\n\r", filename);
#endif
    file = fopen(filename, "w");
    if ( file == NULL ) {
        error("ERROR: Could not open file for writing!\n\r");
        return -1;
    }
    
    // Write header to file
    fprintf(file, "Time,Current\n\r");
        
    // Start timer and begin sampling
    led_state = 0;
#if DEBUG == 1
    printf("Sampling. Do not remove SD card!\n\r");
#endif
    timer.start();
    for ( i = 0; i < NUM_SAMPLES; i++ ) {
        
        // Write data to csv file
        timestamp = timer.read();
        voltage_in = ain * 3.3;
        current = (voltage_in * 1000) / (RS * RL);
        fprintf(file, "%2.2f,%1.3f\n", timestamp, current);
#if DEBUG == 1
        printf("%2.2f, %1.3f\n\r", timestamp, current);
#endif

        // Toggle LED to show logging
        led_state ^= 1;
        status_led = led_state;
        
        // Wait until next sample cycle as defined by sample delay
        while ( (timer.read() - timestamp) < DELAY_MS );
    }
    
    // Close file
#if DEBUG == 1
    printf("Done! Closing file");
#endif
    fclose(file);
    
    // Turn off LED to show done
    status_led = 0;
    
    return 0;
}