/**
 * @file mib2_module.c
 * @brief MIB-II module
 *
 * @section License
 *
 * Copyright (C) 2010-2017 Oryx Embedded SARL. All rights reserved.
 *
 * This file is part of CycloneTCP Open.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * @section Description
 *
 * The second version of the Management Information Base (MIB-II) is used to
 * manage TCP/IP-based hosts. Refer to the following RFCs for complete details:
 * - RFC 1156: MIB for Network Management of TCP/IP-based internets
 * - RFC 1213: MIB for Network Management of TCP/IP-based internets (version 2)
 * - RFC 2863: The Interfaces Group MIB
 * - RFC 4293: MIB for the Internet Protocol (IP)
 * - RFC 4292: IP Forwarding Table MIB
 * - RFC 4022: MIB for the Transmission Control Protocol (TCP)
 * - RFC 4113: MIB for the User Datagram Protocol (UDP)
 * - RFC 3418: MIB for the Simple Network Management Protocol (SNMP)
 * - RFC 4004: Textual Conventions for Internet Network Addresses
 *
 * @author Oryx Embedded SARL (www.oryx-embedded.com)
 * @version 1.7.6
 **/

//Switch to the appropriate trace level
#define TRACE_LEVEL SNMP_TRACE_LEVEL

//Dependencies
#include "core/net.h"
#include "mibs/mib2_module.h"
#include "mibs/mib2_impl.h"
#include "crypto.h"
#include "asn1.h"
#include "oid.h"
#include "debug.h"

//Check TCP/IP stack configuration
#if (MIB2_SUPPORT == ENABLED)


/**
 * @brief MIB-II base
 **/

Mib2Base mib2Base;


/**
 * @brief MIB-II objects
 **/

const MibObject mib2Objects[] =
{
   //System group
   {
      "sysDescr",
      {43, 6, 1, 2, 1, 1, 1},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_ONLY,
#if (MIB2_SYS_DESCR_SIZE > 0)
      &mib2Base.sysGroup.sysDescr,
      &mib2Base.sysGroup.sysDescrLen,
      MIB2_SYS_DESCR_SIZE,
      NULL,
      NULL,
      NULL
#else
      NULL,
      NULL,
      0,
      NULL,
      mib2GetSysDescr,
      NULL
#endif
   },
   {
      "sysObjectID",
      {43, 6, 1, 2, 1, 1, 2},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OBJECT_IDENTIFIER,
      MIB_ACCESS_READ_ONLY,
#if (MIB2_SYS_OBJECT_ID_SIZE > 0)
      &mib2Base.sysGroup.sysObjectID,
      &mib2Base.sysGroup.sysObjectIDLen,
      MIB2_SYS_OBJECT_ID_SIZE,
      NULL,
      NULL,
      NULL
#else
      NULL,
      NULL,
      0,
      NULL,
      mib2GetSysObjectID,
      NULL
#endif
   },
   {
      "sysUpTime",
      {43, 6, 1, 2, 1, 1, 3},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_TIME_TICKS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetSysUpTime,
      NULL
   },
   {
      "sysContact",
      {43, 6, 1, 2, 1, 1, 4},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_WRITE,
#if (MIB2_SYS_CONTACT_SIZE > 0)
      &mib2Base.sysGroup.sysContact,
      &mib2Base.sysGroup.sysContactLen,
      MIB2_SYS_CONTACT_SIZE,
      NULL,
      NULL,
      NULL
#else
      NULL,
      NULL,
      0,
      mib2SetSysContact,
      mib2GetSysContact,
      NULL
#endif
   },
   {
      "sysName",
      {43, 6, 1, 2, 1, 1, 5},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_WRITE,
#if (MIB2_SYS_NAME_SIZE > 0)
      &mib2Base.sysGroup.sysName,
      &mib2Base.sysGroup.sysNameLen,
      MIB2_SYS_NAME_SIZE,
      NULL,
      NULL,
      NULL
#else
      NULL,
      NULL,
      0,
      mib2SetSysName,
      mib2GetSysName,
      NULL
#endif
   },
   {
      "sysLocation",
      {43, 6, 1, 2, 1, 1, 6},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_WRITE,
#if (MIB2_SYS_LOCATION_SIZE > 0)
      &mib2Base.sysGroup.sysLocation,
      &mib2Base.sysGroup.sysLocationLen,
      MIB2_SYS_LOCATION_SIZE,
      NULL,
      NULL,
      NULL
#else
      NULL,
      NULL,
      0,
      mib2SetSysLocation,
      mib2GetSysLocation,
      NULL
#endif
   },
   {
      "sysServices",
      {43, 6, 1, 2, 1, 1, 7},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.sysGroup.sysServices,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   //Interfaces group
   {
      "ifNumber",
      {43, 6, 1, 2, 1, 2, 1},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ifGroup.ifNumber,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   //Interfaces table
   {
      "ifIndex",
      {43, 6, 1, 2, 1, 2, 2, 1, 1},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifDescr",
      {43, 6, 1, 2, 1, 2, 2, 1, 2},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IF_DESCR_SIZE,
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifType",
      {43, 6, 1, 2, 1, 2, 2, 1, 3},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifMtu",
      {43, 6, 1, 2, 1, 2, 2, 1, 4},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifSpeed",
      {43, 6, 1, 2, 1, 2, 2, 1, 5},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_GAUGE32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifPhysAddress",
      {43, 6, 1, 2, 1, 2, 2, 1, 6},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_PHYS_ADDRESS_SIZE,
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifAdminStatus",
      {43, 6, 1, 2, 1, 2, 2, 1, 7},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOperStatus",
      {43, 6, 1, 2, 1, 2, 2, 1, 8},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifLastChange",
      {43, 6, 1, 2, 1, 2, 2, 1, 9},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_TIME_TICKS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifInOctets",
      {43, 6, 1, 2, 1, 2, 2, 1, 10},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifInUcastPkts",
      {43, 6, 1, 2, 1, 2, 2, 1, 11},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifInNUcastPkts",
      {43, 6, 1, 2, 1, 2, 2, 1, 12},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifInDiscards",
      {43, 6, 1, 2, 1, 2, 2, 1, 13},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifInErrors",
      {43, 6, 1, 2, 1, 2, 2, 1, 14},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifInUnknownProtos",
      {43, 6, 1, 2, 1, 2, 2, 1, 15},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOutOctets",
      {43, 6, 1, 2, 1, 2, 2, 1, 16},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOutUcastPkts",
      {43, 6, 1, 2, 1, 2, 2, 1, 17},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOutNUcastPkts",
      {43, 6, 1, 2, 1, 2, 2, 1, 18},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOutDiscards",
      {43, 6, 1, 2, 1, 2, 2, 1, 19},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOutErrors",
      {43, 6, 1, 2, 1, 2, 2, 1, 20},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifOutQLen",
      {43, 6, 1, 2, 1, 2, 2, 1, 21},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_GAUGE32,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(uint32_t),
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
   {
      "ifSpecific",
      {43, 6, 1, 2, 1, 2, 2, 1, 22},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OBJECT_IDENTIFIER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IF_SPECIFIC_SIZE,
      NULL,
      mib2GetIfEntry,
      mib2GetNextIfEntry
   },
#if (IPV4_SUPPORT == ENABLED)
   //IP group
   {
      "ipForwarding",
      {43, 6, 1, 2, 1, 4, 1},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_WRITE,
      &mib2Base.ipGroup.ipForwarding,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipDefaultTTL",
      {43, 6, 1, 2, 1, 4, 2},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_WRITE,
      &mib2Base.ipGroup.ipDefaultTTL,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipInReceives",
      {43, 6, 1, 2, 1, 4, 3},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipInReceives,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipInHdrErrors",
      {43, 6, 1, 2, 1, 4, 4},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipInHdrErrors,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipInAddrErrors",
      {43, 6, 1, 2, 1, 4, 5},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipInAddrErrors,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipForwDatagrams",
      {43, 6, 1, 2, 1, 4, 6},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipForwDatagrams,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipInUnknownProtos",
      {43, 6, 1, 2, 1, 4, 7},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipInUnknownProtos,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipInDiscards",
      {43, 6, 1, 2, 1, 4, 8},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipInDiscards,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipInDelivers",
      {43, 6, 1, 2, 1, 4, 9},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipInDelivers,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipOutRequests",
      {43, 6, 1, 2, 1, 4, 10},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipOutRequests,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipOutDiscards",
      {43, 6, 1, 2, 1, 4, 11},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipOutDiscards,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipOutNoRoutes",
      {43, 6, 1, 2, 1, 4, 12},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipOutNoRoutes,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipReasmTimeout",
      {43, 6, 1, 2, 1, 4, 13},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipReasmTimeout,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipReasmReqds",
      {43, 6, 1, 2, 1, 4, 14},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipReasmReqds,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipReasmOKs",
      {43, 6, 1, 2, 1, 4, 15},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipReasmOKs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipReasmFails",
      {43, 6, 1, 2, 1, 4, 16},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipReasmFails,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipFragOKs",
      {43, 6, 1, 2, 1, 4, 17},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipFragOKs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipFragFails",
      {43, 6, 1, 2, 1, 4, 18},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipFragFails,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "ipFragCreates",
      {43, 6, 1, 2, 1, 4, 19},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipFragCreates,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   //IP address table
   {
      "ipAdEntAddr",
      {43, 6, 1, 2, 1, 4, 20, 1, 1},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_IP_ADDRESS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IP_ADDRESS_SIZE,
      NULL,
      mib2GetIpAddrEntry,
      mib2GetNextIpAddrEntry
   },
   {
      "ipAdEntIfIndex",
      {43, 6, 1, 2, 1, 4, 20, 1, 2},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIpAddrEntry,
      mib2GetNextIpAddrEntry
   },
   {
      "ipAdEntNetMask",
      {43, 6, 1, 2, 1, 4, 20, 1, 3},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_IP_ADDRESS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IP_ADDRESS_SIZE,
      NULL,
      mib2GetIpAddrEntry,
      mib2GetNextIpAddrEntry
   },
   {
      "ipAdEntBcastAddr",
      {43, 6, 1, 2, 1, 4, 20, 1, 4},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIpAddrEntry,
      mib2GetNextIpAddrEntry
   },
   {
      "ipAdEntReasmMaxSize",
      {43, 6, 1, 2, 1, 4, 20, 1, 5},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetIpAddrEntry,
      mib2GetNextIpAddrEntry
   },
   //IP address translation table
   {
      "ipNetToMediaIfIndex",
      {43, 6, 1, 2, 1, 4, 22, 1, 1},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_WRITE,
      NULL,
      NULL,
      sizeof(int32_t),
      mib2SetIpNetToMediaEntry,
      mib2GetIpNetToMediaEntry,
      mib2GetNextIpNetToMediaEntry
   },
   {
      "ipNetToMediaPhysAddress",
      {43, 6, 1, 2, 1, 4, 22, 1, 2},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_OCTET_STRING,
      MIB_ACCESS_READ_WRITE,
      NULL,
      NULL,
      MIB2_PHYS_ADDRESS_SIZE,
      mib2SetIpNetToMediaEntry,
      mib2GetIpNetToMediaEntry,
      mib2GetNextIpNetToMediaEntry
   },
   {
      "ipNetToMediaNetAddress",
      {43, 6, 1, 2, 1, 4, 22, 1, 3},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_IP_ADDRESS,
      MIB_ACCESS_READ_WRITE,
      NULL,
      NULL,
      MIB2_IP_ADDRESS_SIZE,
      mib2SetIpNetToMediaEntry,
      mib2GetIpNetToMediaEntry,
      mib2GetNextIpNetToMediaEntry
   },
   {
      "ipNetToMediaType",
      {43, 6, 1, 2, 1, 4, 22, 1, 4},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_WRITE,
      NULL,
      NULL,
      sizeof(int32_t),
      mib2SetIpNetToMediaEntry,
      mib2GetIpNetToMediaEntry,
      mib2GetNextIpNetToMediaEntry
   },
   //Additional IP objects
   {
      "ipRoutingDiscards",
      {43, 6, 1, 2, 1, 4, 23},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.ipGroup.ipRoutingDiscards,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   //ICMP group
   {
      "icmpInMsgs",
      {43, 6, 1, 2, 1, 5, 1},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInMsgs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInErrors",
      {43, 6, 1, 2, 1, 5, 2},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInErrors,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInDestUnreachs",
      {43, 6, 1, 2, 1, 5, 3},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInDestUnreachs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInTimeExcds",
      {43, 6, 1, 2, 1, 5, 4},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInTimeExcds,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInParmProbs",
      {43, 6, 1, 2, 1, 5, 5},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInParmProbs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInSrcQuenchs",
      {43, 6, 1, 2, 1, 5, 6},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInSrcQuenchs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInRedirects",
      {43, 6, 1, 2, 1, 5, 7},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInRedirects,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInEchos",
      {43, 6, 1, 2, 1, 5, 8},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInEchos,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInEchoReps",
      {43, 6, 1, 2, 1, 5, 9},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInEchoReps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInTimestamps",
      {43, 6, 1, 2, 1, 5, 10},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInTimestamps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInTimestampReps",
      {43, 6, 1, 2, 1, 5, 11},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInTimestampReps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInAddrMasks",
      {43, 6, 1, 2, 1, 5, 12},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInAddrMasks,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpInAddrMaskReps",
      {43, 6, 1, 2, 1, 5, 13},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpInAddrMaskReps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutMsgs",
      {43, 6, 1, 2, 1, 5, 14},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutMsgs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutErrors",
      {43, 6, 1, 2, 1, 5, 15},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutErrors,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutDestUnreachs",
      {43, 6, 1, 2, 1, 5, 16},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutDestUnreachs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutTimeExcds",
      {43, 6, 1, 2, 1, 5, 17},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutTimeExcds,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutParmProbs",
      {43, 6, 1, 2, 1, 5, 18},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutParmProbs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutSrcQuenchs",
      {43, 6, 1, 2, 1, 5, 19},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutSrcQuenchs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutRedirects",
      {43, 6, 1, 2, 1, 5, 20},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutRedirects,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutEchos",
      {43, 6, 1, 2, 1, 5, 21},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutEchos,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutEchoReps",
      {43, 6, 1, 2, 1, 5, 22},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutEchoReps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutTimestamps",
      {43, 6, 1, 2, 1, 5, 23},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutTimestamps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutTimestampReps",
      {43, 6, 1, 2, 1, 5, 24},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutTimestampReps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutAddrMasks",
      {43, 6, 1, 2, 1, 5, 25},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutAddrMasks,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "icmpOutAddrMaskReps",
      {43, 6, 1, 2, 1, 5, 26},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.icmpGroup.icmpOutAddrMaskReps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
#endif
#if (TCP_SUPPORT == ENABLED)
   //TCP group
   {
      "tcpRtoAlgorithm",
      {43, 6, 1, 2, 1, 6, 1},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpRtoAlgorithm,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpRtoMin",
      {43, 6, 1, 2, 1, 6, 2},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpRtoMin,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpRtoMax",
      {43, 6, 1, 2, 1, 6, 3},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpRtoMax,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpMaxConn",
      {43, 6, 1, 2, 1, 6, 4},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpMaxConn,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpActiveOpens",
      {43, 6, 1, 2, 1, 6, 5},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpActiveOpens,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpPassiveOpens",
      {43, 6, 1, 2, 1, 6, 6},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpPassiveOpens,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpAttemptFails",
      {43, 6, 1, 2, 1, 6, 7},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpAttemptFails,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpEstabResets",
      {43, 6, 1, 2, 1, 6, 8},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpEstabResets,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpCurrEstab",
      {43, 6, 1, 2, 1, 6, 9},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_GAUGE32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpCurrEstab,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpInSegs",
      {43, 6, 1, 2, 1, 6, 10},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpInSegs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpOutSegs",
      {43, 6, 1, 2, 1, 6, 11},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpOutSegs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpRetransSegs",
      {43, 6, 1, 2, 1, 6, 12},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpRetransSegs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   //TCP connection table
   {
      "tcpConnState",
      {43, 6, 1, 2, 1, 6, 13, 1, 1},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_WRITE,
      NULL,
      NULL,
      sizeof(int32_t),
      mib2SetTcpConnEntry,
      mib2GetTcpConnEntry,
      mib2GetNextTcpConnEntry
   },
   {
      "tcpConnLocalAddress",
      {43, 6, 1, 2, 1, 6, 13, 1, 2},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_IP_ADDRESS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IP_ADDRESS_SIZE,
      NULL,
      mib2GetTcpConnEntry,
      mib2GetNextTcpConnEntry
   },
   {
      "tcpConnLocalPort",
      {43, 6, 1, 2, 1, 6, 13, 1, 3},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetTcpConnEntry,
      mib2GetNextTcpConnEntry
   },
   {
      "tcpConnRemAddress",
      {43, 6, 1, 2, 1, 6, 13, 1, 4},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_IP_ADDRESS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IP_ADDRESS_SIZE,
      NULL,
      mib2GetTcpConnEntry,
      mib2GetNextTcpConnEntry
   },
   {
      "tcpConnRemPort",
      {43, 6, 1, 2, 1, 6, 13, 1, 5},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetTcpConnEntry,
      mib2GetNextTcpConnEntry
   },
   //Additional TCP objects
   {
      "tcpInErrs",
      {43, 6, 1, 2, 1, 6, 14},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpInErrs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpOutRsts",
      {43, 6, 1, 2, 1, 6, 15},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpOutRsts,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpHCInSegs",
      {43, 6, 1, 2, 1, 6, 17},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER64,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpHCInSegs,
      NULL,
      sizeof(uint64_t),
      NULL,
      NULL,
      NULL
   },
   {
      "tcpHCOutSegs",
      {43, 6, 1, 2, 1, 6, 18},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER64,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.tcpGroup.tcpHCOutSegs,
      NULL,
      sizeof(uint64_t),
      NULL,
      NULL,
      NULL
   },
#endif
#if (UDP_SUPPORT == ENABLED)
   //UDP group
   {
      "udpInDatagrams",
      {43, 6, 1, 2, 1, 7, 1},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.udpGroup.udpInDatagrams,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "udpNoPorts",
      {43, 6, 1, 2, 1, 7, 2},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.udpGroup.udpNoPorts,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "udpInErrors",
      {43, 6, 1, 2, 1, 7, 3},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.udpGroup.udpInErrors,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "udpOutDatagrams",
      {43, 6, 1, 2, 1, 7, 4},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.udpGroup.udpOutDatagrams,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   //UDP listener table
   {
      "udpLocalAddress",
      {43, 6, 1, 2, 1, 7, 5, 1, 1},
      9,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_IP_ADDRESS,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      MIB2_IP_ADDRESS_SIZE,
      NULL,
      mib2GetUdpEntry,
      mib2GetNextUdpEntry
   },
   {
      "udpLocalPort",
      {43, 6, 1, 2, 1, 7, 5, 1, 2},
      9,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_ONLY,
      NULL,
      NULL,
      sizeof(int32_t),
      NULL,
      mib2GetUdpEntry,
      mib2GetNextUdpEntry
   },
   //Additional UDP objects
   {
      "udpHCInDatagrams",
      {43, 6, 1, 2, 1, 7, 8},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER64,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.udpGroup.udpHCInDatagrams,
      NULL,
      sizeof(uint64_t),
      NULL,
      NULL,
      NULL
   },
   {
      "udpHCOutDatagrams",
      {43, 6, 1, 2, 1, 7, 9},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER64,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.udpGroup.udpHCOutDatagrams,
      NULL,
      sizeof(uint64_t),
      NULL,
      NULL,
      NULL
   },
#endif
   //SNMP group
   {
      "snmpInPkts",
      {43, 6, 1, 2, 1, 11, 1},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInPkts,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutPkts",
      {43, 6, 1, 2, 1, 11, 2},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutPkts,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInBadVersions",
      {43, 6, 1, 2, 1, 11, 3},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInBadVersions,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInBadCommunityNames",
      {43, 6, 1, 2, 1, 11, 4},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInBadCommunityNames,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInBadCommunityUses",
      {43, 6, 1, 2, 1, 11, 5},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInBadCommunityUses,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInASNParseErrs",
      {43, 6, 1, 2, 1, 11, 6},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInASNParseErrs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInTooBigs",
      {43, 6, 1, 2, 1, 11, 8},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInTooBigs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInNoSuchNames",
      {43, 6, 1, 2, 1, 11, 9},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInNoSuchNames,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInBadValues",
      {43, 6, 1, 2, 1, 11, 10},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInBadValues,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInReadOnlys",
      {43, 6, 1, 2, 1, 11, 11},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInReadOnlys,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInGenErrs",
      {43, 6, 1, 2, 1, 11, 12},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInGenErrs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInTotalReqVars",
      {43, 6, 1, 2, 1, 11, 13},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInTotalReqVars,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInTotalSetVars",
      {43, 6, 1, 2, 1, 11, 14},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInTotalSetVars,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInGetRequests",
      {43, 6, 1, 2, 1, 11, 15},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInGetRequests,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInGetNexts",
      {43, 6, 1, 2, 1, 11, 16},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInGetNexts,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInSetRequests",
      {43, 6, 1, 2, 1, 11, 17},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInSetRequests,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInGetResponses",
      {43, 6, 1, 2, 1, 11, 18},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInGetResponses,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpInTraps",
      {43, 6, 1, 2, 1, 11, 19},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpInTraps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutTooBigs",
      {43, 6, 1, 2, 1, 11, 20},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutTooBigs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutNoSuchNames",
      {43, 6, 1, 2, 1, 11, 21},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutNoSuchNames,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutBadValues",
      {43, 6, 1, 2, 1, 11, 22},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutBadValues,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutGenErrs",
      {43, 6, 1, 2, 1, 11, 24},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutGenErrs,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutGetRequests",
      {43, 6, 1, 2, 1, 11, 25},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutGetRequests,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutGetNexts",
      {43, 6, 1, 2, 1, 11, 26},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutGetNexts,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutSetRequests",
      {43, 6, 1, 2, 1, 11, 27},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutSetRequests,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutGetResponses",
      {43, 6, 1, 2, 1, 11, 28},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutGetResponses,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpOutTraps",
      {43, 6, 1, 2, 1, 11, 29},
      7,
      ASN1_CLASS_APPLICATION,
      MIB_TYPE_COUNTER32,
      MIB_ACCESS_READ_ONLY,
      &mib2Base.snmpGroup.snmpOutTraps,
      NULL,
      sizeof(uint32_t),
      NULL,
      NULL,
      NULL
   },
   {
      "snmpEnableAuthenTraps",
      {43, 6, 1, 2, 1, 11, 30},
      7,
      ASN1_CLASS_UNIVERSAL,
      ASN1_TYPE_INTEGER,
      MIB_ACCESS_READ_WRITE,
      &mib2Base.snmpGroup.snmpEnableAuthenTraps,
      NULL,
      sizeof(int32_t),
      NULL,
      NULL,
      NULL
   }
};


/**
 * @brief MIB-II module
 **/

const MibModule mib2Module =
{
   mib2Objects,
   arraysize(mib2Objects),
   mib2Init,
   mib2Lock,
   mib2Unlock
};

#endif

