// Note!!!! This software is under a beerware license, 
//          please send a beer to clarke.sam1@gmail.com 
//          via paypal if you find this code useful.


#include "Plan13.h" 
#include "GPS.h"
#include "mbed.h"

#define ONEPPM              1.0e-6
#define ONEPPTM             1.0e-7

struct Target{
    int   YEAR;
    float EPOCH;
    float INCLINATION;
    float RAAN;
    float ECCENTRICITY;
    float ARGUMENT_PEDIGREE;
    float MEAN_ANOMALY;
    float MEAN_MOTION;
    float TIME_MOTION_D;
    float EPOCH_ORBIT;
    float azimuth;
    float elevation;
};

char *astra2f_tle = 
    "ASTRA 2F"                                            
    "1 38778U 12051A   14010.00000000  .00000143  00000-0  00000+0 0  2381"
    "2 38778 000.0670 267.2510 0004011 341.2500 249.1500 01.00276604  4755";

void target_tle(int year, char *tle, struct Target *t);

void calculate(Plan13& p, GPS& g, struct Target *t);

void target_data(int yr,        float epoch,       float inc,  float raan,
                float ecc,     float arg,         float anom, float mmotion,
                float tmotion, float epoch_orbit, struct Target *t);

Plan13 p;
GPS gps(p9,p10);
Serial pc(USBTX, USBRX);

int main() 
{ 
    struct Target ASTRA2F;                    // Create a target object for ASTRA2F
    target_tle(2014, astra2f_tle, &ASTRA2F);  // Parse a string TLE into the object, or set the object manually (below)
    
    // NOAA 19
    //1 33591U 09005A   14018.50690348  .00000100  00000-0  79451-4 0   929
    //2 33591 098.9294 324.1290 0014157 178.0460 233.2637 14.11623746254876
    struct Target NOAA19;
    target_data(2014, 18.50690348, 98.9294, 324.1290, 79451,
               178.0460, 233.2637, 14.1162376, 0.00000100, 25487, &NOAA19);            
               
    gps.Init();
    pc.baud(115200);              
    while (1) 
    {
        wait(1);
        if(gps.parseData()) {
            calculate(p, gps, &ASTRA2F);
            pc.printf("GPS Position--------[ %f, %f ]\n\n", gps.latitude, gps.longitude);
            pc.printf("ASTRA2F Azimuth-----[ %f ]\n", ASTRA2F.azimuth);  
            pc.printf("ASTRA2F Elevation---[ %f ]\n\n", ASTRA2F.elevation);
            calculate(p, gps, &NOAA19);
            pc.printf("NOAA19 Azimuth -----[ %f ]\n", NOAA19.azimuth);  
            pc.printf("NOAA19 Elevation----[ %f ]\n\n", NOAA19.elevation);
        }else {
            pc.printf("Aquiring Satellites...\n");
        }
    }
     
}

void calculate(Plan13& p, GPS& g, struct Target *t )
{
    p.setTime(g.year,g.month,g.day,g.hours, g.minutes, g.seconds); //YMDHMS
    p.setElements(t->YEAR,
                  t->EPOCH,
                  t->INCLINATION, 
                  t->RAAN,
                  t->ECCENTRICITY*ONEPPTM, 
                  t->ARGUMENT_PEDIGREE,
                  t->MEAN_ANOMALY, 
                  t->MEAN_MOTION,
                  t->TIME_MOTION_D, 
                  t->EPOCH_ORBIT+ONEPPM,
                  0);
    p.setLocation(g.longitude,g.latitude,g.altitude);
    p.calculate();
    t->azimuth = p.AZ;
    t->elevation = p.EL;
}

void target_data(int yr,        float epoch,       float inc,  float raan,
                 float ecc,     float arg,         float anom, float mmotion,
                 float tmotion, float epoch_orbit, struct Target *t)
{
    t->YEAR = yr;
    t->EPOCH = epoch;
    t->INCLINATION = inc;
    t->RAAN = raan;
    t->ECCENTRICITY = ecc; 
    t->ARGUMENT_PEDIGREE = arg;
    t->MEAN_ANOMALY = anom;
    t->MEAN_MOTION = mmotion;
    t->TIME_MOTION_D = tmotion; 
    t->EPOCH_ORBIT = epoch_orbit;
}

void target_tle(int year, char *tle, struct Target *t){
    t->YEAR = year;
    sscanf(tle, 
        "%*s %*s %*s %*s  %*c%*c%*c%f  %f  %*s  %f%*c%*c %*c %*s %*s %f %f "
        "%*s %f %f %f%*c%f", 
        &t->EPOCH, &t->TIME_MOTION_D, &t->ECCENTRICITY, &t->INCLINATION, &t->RAAN,
        &t->ARGUMENT_PEDIGREE, &t->MEAN_ANOMALY, &t->MEAN_MOTION, &t->EPOCH_ORBIT);
}