/***************************************************************************//**
 * @file main.cpp
 *******************************************************************************
 * @section License
 * <b>(C) Copyright 2017 Silicon Labs, http://www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ******************************************************************************/

#include <string>
#include "mbed.h"
#include "BMP280/BMP280.h"
#include "AMS_CCS811.h"
#include "Si7021.h"
#include "Si7210.h"
#include "Si1133.h"
#include "ICM20648.h"

#include "SPI_MIP16.h"
#include "data.h"
#include "HGP23x29.h"
#include "HGP15x19.h"


/* Turn on power supply to ENV sensor suite */
DigitalOut env_en(PF9, 1);
/* Turn on power to CCS811 sensor */
DigitalOut ccs_en(PF14, 1);
/* Set CCS_WAKE pin to output */
DigitalOut ccs_wake(PF15, 1);
/* Turn on power to hall effect sensor */
DigitalOut hall_en(PB10, 1);
/* Turn on power to IMU */
DigitalOut imu_en(PF8, 1);

I2C ccs_i2c(PB6, PB7);
I2C hall_i2c(PB8, PB9);
I2C env_i2c(PC4, PC5);

// MIP
WatchDisp WD(PK0,PK2,PF7,PA5); //PTC12);  // mosi,miso,clk,cs,disp,power(EXTMODE)
DigitalOut _lcd_disp(PA8);    // DISP
DigitalOut _lcd_extmod(PF6);  // EXTMODE L=SERIAL, H=EXTCOMIN Signal
DigitalOut _lcd_extcom(PA9);  // Togle

uint8_t ccolor_table[] = {Black,Gray,Blue   ,LightBlue   ,Green ,LightGreen ,Cyan     ,LightCyan,
                          Red  ,pink,Magenta,LightMagenta,Yellow,LightYellow,LightGray,White};
DigitalOut _pk1(PK1,1);  // QSPI Flash CS Disable

bool  if16 = 0;
bool  ifMargeTXT = 0;
int  width =176;
int  height=176;
///////////////////////////////////////
int disp_logo(void)
{
    int RGB;
    uint8_t R8,G8,B8;
    uint8_t *cptr;
    //for(int i=0; i< 54 ; i++) fscanf(fp,"%c",DUMMY);  // Discard Header 54bytes
    cptr = (uint8_t *)&Title_bmp[54];
    for(int y=height-1; y>=0; y--) {
        for(int x=0; x< width; x++) {    //24bit color  B 8bit -> G 8bit -> R 8bit
            B8 = *cptr++;
            G8 = *cptr++;
            R8 = *cptr++;
            RGB =  RGB8(R8,G8,B8);    //6bit(8bit) MIP  MASK 0000 1110
            WD.pixel(x,y,RGB);
        }
        //if( y!=0)  // The last data column doesn't need padding
        //for(int x=(width*3)%4; (x%4 !=0); x++) fscanf(fp, "%c",DUMMY);    // 4byte boundery for every column
    }
}

int lcd_init(void)
{
    int i;
    // Power On Sequece
//  _lcd_reset  = 0;
//  _lcd_reset  = 1;
    _lcd_extmod = 0;

    WD.setWH(width,height);     // Default   X/Y-00
    bool  if16 = 1;
    WD.set16col(if16);          // 0= RGB, 1=RGBW
    WD.setmarge(1);
    WD.background(Cyan);
    WD.clsBUF();
    _lcd_disp = 1;              // 
    //WD.fillrect(i*(width/16), height/2+1, (i+1)*(width/16)-1, height  , ccolor_table[15-i]);
    //WD.writeDISP();         
    
    //WD.set_font((unsigned char*) HGP23x29);
    //WD.locate(  0, 43);WD.foreground(Blue); WD.printf("Environmental");
    //WD.locate(  0, 83);WD.foreground(Blue); WD.printf("   Monitor");
    //WD.locate(120,140);WD.foreground(Red ); WD.printf("JDI");
    disp_logo();
    WD.writeDISP();               
    _lcd_extmod = 1;   // 1=Pin,0=Serial
    WD.background(Cyan);
    return 0;
/**/    
}

///////////////////////////////////////////////
int main() {
    uint32_t eco2, tvoc;
    float pressure, temperature2;
    int32_t temperature;
    uint32_t humidity;
    float light, uv;
    float acc_x, acc_y, acc_z, gyr_x, gyr_y, gyr_z, temperature3;

    bool lightsensor_en = true;
    bool gassensor_en = true;
    bool hallsensor_en = true;
    bool rhtsensor_en = true;
    bool pressuresensor_en = true;
    bool imu_en = false;

    _pk1=1;
    lcd_init();
    ccs_en = 0; wait_ms(100);
    ccs_en = 1;
    wait_ms(2000);

    /* Initialize air quality sensor */
    AMS_CCS811* gasSensor = new AMS_CCS811(&ccs_i2c, PF15);
    if(!gasSensor->init()) {
        printf("Failed CCS811 init\r\n");
        gassensor_en = false;
    } else {
        if(!gasSensor->mode(AMS_CCS811::TEN_SECOND)) {
            printf("Failed to set CCS811 mode\r\n");
            gassensor_en = false;
        }
    }
    
    wait_ms(10);

    /* Initialize barometer and RHT */
    BMP280* pressureSensor = new BMP280(env_i2c);
    Si7021* rhtSensor = new Si7021(PC4, PC5);

    printf("\r\n\r\nHello Thunderboard Sense 2...\r\n");
    
    /* Check if hall sensor is alive */
    silabs::Si7210* hallSensor = new silabs::Si7210(&hall_i2c);
    uint8_t id;
    hallSensor->wakeup();
    hallSensor->readRegister(SI72XX_HREVID, &id);
    printf("Hall ID: %d\r\n", id);

    /* Initialize light sensor */
    Si1133* lightSensor = new Si1133(PC4, PC5);

    if(!lightSensor->open()) {
        printf("Something is wrong with Si1133, disabling...\n");
        lightsensor_en = false;
    } else printf("Si1133 available\n\r");

    /* Initialize the IMU*/
    printf("ICM20648 Start \n\r");

    ICM20648* imu = new ICM20648(PC0, PC1, PC2, PC3, PF12);
    printf(" ICM20648 OK\n\r");

    if (imu != 0){
        if(!imu->open()) {
            printf("Something is wrong with ICM20648, disabling...\n");
            imu_en = false;
        } else printf("ICM20648 available\n\r");
    }
    WD.foreground(Red);
    WD.set_font((unsigned char*) HGP15x19);   
#define WD_OFFSET 10
    do {
        WD.clsBUF();
        printf("----------------------------------------------\r\n");
        /* Measure temperature and humidity */
        if(rhtsensor_en) {
            rhtSensor->measure();
            rhtSensor->measure();
            temperature = rhtSensor->get_temperature();
            humidity = rhtSensor->get_humidity();

            printf("temperature and humidity(Si7021):\r\n");
            printf("temperature: %ld.%03ld degC\r\n", temperature/1000, abs(temperature%1000));
            printf("humidity   : %ld.%03ld %%\r\n", humidity/1000, humidity%1000);
            WD.locate( 2,0*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("Temp.");
            WD.locate(55,0*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%ld.%03ld degC", temperature/1000, abs(temperature%1000));
            WD.locate( 2,1*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("Humi.");
            WD.locate(55,1*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%ld.%03ld %%", humidity/1000, humidity%1000);
        }
        
        /* Measure barometric pressure */
        if(pressuresensor_en) {
            temperature2 = pressureSensor->getTemperature();
            pressure = pressureSensor->getPressure();

            printf("pressure(BMP280):\r\n");
            printf("pressure   : %.2f bar\r\n", pressure);
            printf("temperature: %.2f degC\r\n", temperature2);
            WD.locate( 2,2*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("Pres.");
            WD.locate(55,2*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%.2f bar\r\n", pressure);
        }

        /* Measure air quality */
        if(gassensor_en) {
            gasSensor->has_new_data();
            eco2 = gasSensor->co2_read();
            tvoc = gasSensor->tvoc_read();

            printf("air quality(CCS811):\r\n");
            printf("CO2: %ld ppm\r\n", eco2);
            printf("VoC: %ld ppb\r\n", tvoc);
            WD.locate( 2,3*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("CO2");
            WD.locate(55,3*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%ld ppm", eco2);
            WD.locate( 2,4*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("VoC");
            WD.locate(55,4*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%ld ppb", tvoc);
        }
        
        /* measure HALL */
        if(hallsensor_en) {
            hallSensor->measureOnce();
            printf("HALL(Si7210):\r\n");
            printf("T: %d.%02d degC\r\n", hallSensor->getTemperature()/1000, abs(hallSensor->getTemperature()%1000));
            printf("F: %i.%03d mT\r\n", (int)(hallSensor->getFieldStrength()/1000), abs(hallSensor->getFieldStrength() % 1000));
            WD.locate( 2,5*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("Magn.");
            WD.locate(55,5*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%i.%03d mT\r\n", (int)(hallSensor->getFieldStrength()/1000), abs(hallSensor->getFieldStrength() % 1000));
        }
        
        /* measure light */
        if(lightsensor_en) {
            lightSensor->get_light_and_uv(&light, &uv);

            printf("Light(BMP280):\r\n");
            printf("Light: %.2f lux\r\n", light);
            printf("UV   : %.2f    \r\n", uv);
            WD.locate( 2,6*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("Lumi.");
            WD.locate(55,6*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%.2f lux", light);
            WD.locate( 2,7*20+WD_OFFSET);WD.foreground(Blue ); WD.printf("UV");
            WD.locate(55,7*20+WD_OFFSET);WD.foreground(Red  ); WD.printf("%.2f", uv);
        }

        /* measure imu */
        if(imu_en) {
            imu->get_temperature(&temperature3);
            imu->get_accelerometer(&acc_x, &acc_y, &acc_z);
            imu->get_gyroscope(&gyr_x, &gyr_y, &gyr_z);

            printf("ICM20648:\r\n");
            printf("TEMP: %.2f degC\r\n", temperature3);
            printf("Acc : %.2f %.2f %.2f\r\n", acc_x, acc_y, acc_z);
            printf("Gyro: %.2f %.2f %.2f\r\n", gyr_x, gyr_y, gyr_z);
            WD.locate(2,8*20+WD_OFFSET); WD.printf("Acc : %.2f %.2f %.2f\r\n", acc_x, acc_y, acc_z);
            WD.locate(2,9*20+WD_OFFSET); WD.printf("Gyro: %.2f %.2f %.2f\r\n", gyr_x, gyr_y, gyr_z);
        }
        WD.writeDISP();        
        _lcd_extcom = !_lcd_extcom;
        wait(1);
    } while(1);
}
