#include "A3921_2p.h"
#include "encoder.h"
#include "KyubeBoard.h"
#include "PID.h"


//-----------インスタンスを生成-----------//
Serial pc(USBTX, USBRX, 9600);
//モータドライバ
A3921_2p m_left(M1_PWML, M1_PHASE);
A3921_2p m_right(M2_PWML, M2_PHASE);

//エンコーダ
encoder enc_left(M1_ENCODER_A, M1_ENCODER_B, 7);
encoder enc_right(M2_ENCODER_A, M2_ENCODER_B, 7);

//アナログ
AnalogIn line_left1(ANALOG_LEFT_1);
AnalogIn line_right1(ANALOG_RIGHT_1);

AnalogIn line_left2(ANALOG_LEFT_2);
AnalogIn line_right2(ANALOG_RIGHT_2);

//タクトスイッチ
DigitalIn button(BUTTON);

//LED
BusOut led(LED1);

//タイマ割り込み
Ticker flipper;

//パラメータ関係

//PID
PID velo_l(VELO_KP, VELO_KI, VELO_KD, SAMPLING_TIME);
PID velo_r(VELO_KP, VELO_KI, VELO_KD, SAMPLING_TIME);
PID line_pid(LINE_KP, LINE_KI, LINE_KD, SAMPLING_TIME);

/*
OK
4.8, 0.0, 0.075, 3000f, 5.6f 急カーブに対応できた．振動でかい
*/

//-------------グローバル変数-------------//
//目標速度
float target_v_l = 0.0;
float target_v_r = 0.0;

//-------------初期設定関数-------------//
void init()
{
    //モータ
    m_left.SetPeriod_ms(PULSE_PERIOD_MS);   //周期を設定
    m_left.SetReverse(true);                //m_leftの回転方向指定
    m_left.SetEnable(false);                //m_leftを駆動禁止

    m_right.SetPeriod_ms(PULSE_PERIOD_MS);  //周期を設定
    m_right.SetReverse(false);              //m_rightの回転方向指定
    m_right.SetEnable(false);               //m_rightを駆動禁止

    //エンコーダ
    enc_left.SetReverse(true);   //enc_m_leftの回転方向指定
    enc_right.SetReverse(false); //enc_m_rightの回転方向指定
    
    //ボタンをプルアップ
    button.mode(PullUp);    
    
    //PID関係の制限
    velo_l.EnableCtrlLimit(-1000.0f, 1000.0f);//モータのDuty比を制限
    velo_r.EnableCtrlLimit(-1000.0f, 1000.0f);
    velo_l.EnableInteLimit(-2000.0f, 2000.0f);//積分値を制限
    velo_r.EnableInteLimit(-2000.0f, 2000.0f);
    
    //ローパスフィルタを有効化
    line_pid.EnableLPF(0.8);
}

//-------------ラインセンサ関数-------------//
void line()
{
    //ラインセンサの差を計算
    float diff_line_1 = (float)(line_left1.read_u16()>>6) - (line_right1.read_u16()>>6);
    float diff_line_2 = (float)(line_left2.read_u16()>>6) - (line_right2.read_u16()>>6);
    
    //偏差データを加工
    diff_line_1 = fabs(diff_line_1)<LINE_ERROR? 0.0f: diff_line_1;
    diff_line_2 = fabs(diff_line_2)<LINE_ERROR? 0.0f: diff_line_2*LINE_GAIN;
    
    //操作量を計算
    float line_ctrl = line_pid.Controller(0.0, (float)(diff_line_1+diff_line_2));

    //目標速度計算
    target_v_l = DEFAULT_VELOCITY - line_ctrl;
    target_v_r = DEFAULT_VELOCITY + line_ctrl;

}

//-------------モータ制御関数-------------//
void motor()
{
    //速度計算
    float v_l = (float)(enc_left.GetCount()/SAMPLING_TIME);
    float v_r = (float)(enc_right.GetCount()/SAMPLING_TIME);

    enc_left.ResetCount();
    enc_right.ResetCount();

    //速度PID
    float duty_l = velo_l.Controller(target_v_l, v_l);
    float duty_r = velo_r.Controller(target_v_r, v_r);

    m_left.SetDuty(duty_l);
    m_right.SetDuty(duty_r);
}

//-------------割り込み制御関数-------------//
void flip()
{
    line();
    motor();
}



int main()
{
    //初期化
    pc.printf("initializing\r\n");
    init();

    int button_count=0;

    while(button_count<5) { //ボタンが長押しされるまで待機
        int l1 = line_left1.read_u16()>>6;
        int r1 = line_right1.read_u16()>>6;
        int l2 = line_left2.read_u16()>>6;
        int r2 = line_right2.read_u16()>>6;
        int diff1 = l1-r1;
        int diff2 = l2-r2;
        
        //ボタンが連続で押されていたらカウントをインクリメント
        button_count = !button? button_count+1: 0;
        led = button_count;
        pc.printf("%4d, %4d, %4d, %4d | diff1:%4d, diff2:%4d, button_count:%d\n\r", l2, l1, r1, r2, diff1, diff2, button_count);
        wait(0.3);
    }
    while(!button){};
    
    //スタート準備
    pc.printf("line-trace leady\n\r");
    led=1;      //LEDを点灯
    wait(2);    //2秒待機
    pc.printf("line-trace start!!\n\r");

    led=0;      //LED消灯

    //モータ駆動許可
    m_left.SetEnable(true);
    m_right.SetEnable(true);

    //エンコーダ測定開始
    enc_left.StartCount();
    enc_right.StartCount();

    //割り込み開始
    flipper.attach(&flip, SAMPLING_TIME);
    
    button_count=0;
    while(button_count<3) {
        button_count = !button? button_count+1: 0;
        led = !led; //LED点滅
        wait(0.1);
    }
}
