#ifndef __PID__
#define __PID__

class PID
{
public:
    PID(float _kp, float _ki, float _kd, float _period=0.001, float _target=0.0)
    {
        //各ゲイン
        kp = _kp;
        ki = _ki;
        kd = _kd;

        //周期
        period = _period;

        //目標値
        target = _target;

        //各種変数初期化
        pre_diff = 0.0;
        i_val = 0.0;
        lpf_flag = false;
        ctrl_limit_flag = false;
        inte_limit_flag = false;

        lpf_weight = 0.0f;

        ctrl_limit_max = ctrl_limit_min = 0.0f;
        inte_limit_max = inte_limit_min = 0.0f;
    };

    ~PID() {};

    void SetPeriod(float _period)
    {
        period = _period;
    };

    void SetTarget(float _target)
    {
        target = _target;
    };

    //LPFを有効化　xはLPFの係数
    void EnableLPF(float x)
    {
        x = (x>1.0f)? 1.0f: x;
        x = (x<0.0f)? 0.0f: x;
        lpf_weight = (x<=0.5f)? (1.0f-x): x;
        lpf_flag = true;
    }

    //LPFを無効化
    void DisEnableLPF()
    {
        lpf_flag = false;
    }

    //操作量の制限を有効化
    void EnableCtrlLimit(float l1, float l2)
    {
        ctrl_limit_max = (l1>l2)? l1: l2;
        ctrl_limit_min = (l1<l2)? l1: l2;
        ctrl_limit_flag = true;
    }

    //操作量の制限を無効化
    void DisEnableCtrlLimit()
    {
        ctrl_limit_flag = false;
    }

    //積分値の制限を有効化
    void EnableInteLimit(float l1, float l2)
    {
        inte_limit_max = (l1>l2)? l1: l2;
        inte_limit_min = (l1<l2)? l1: l2;
        inte_limit_flag = true;
    }

    //積分値の制限を無効化
    void DisEnableInteLimit()
    {
        inte_limit_flag = false;
    }


    void Reset()
    {
        //各種変数初期化
        pre_diff = 0.0;
        i_val = 0.0;
    };

    float Controller(float present)
    {
        //偏差を計算
        float diff = present - target;

        //LPFを通す
        diff = lpf_flag? (diff*(1.0f-lpf_weight) + pre_diff*lpf_weight): diff;

        //数値微分
        float d_val = (diff - pre_diff)/period;
        pre_diff = diff;

        //数値積分
        i_val += (diff + pre_diff) * period *0.5f;

        //積分値が制限を超えてないかチェック
        if(inte_limit_flag) {
            i_val = i_val>inte_limit_max? inte_limit_max: i_val;
            i_val = i_val<inte_limit_min? inte_limit_min: i_val;
        }

        //操作量を計算
        float ctrl = kp * diff + kd * d_val + ki * i_val;

        //操作量が制限を超えてないかチェック
        if(ctrl_limit_flag) {
            ctrl = ctrl>ctrl_limit_max? ctrl_limit_max: ctrl;
            ctrl = ctrl<ctrl_limit_min? ctrl_limit_min: ctrl;
        }

        return ctrl;
    };

    float Controller(float _target, float present)
    {
        //偏差を計算
        float diff = present - _target;

        
        //LPFを通す
        diff = lpf_flag? (diff*(1.0f-lpf_weight) + pre_diff*lpf_weight): diff;

        //数値微分
        float d_val = (diff - pre_diff)/period;
        pre_diff = diff;

        //数値積分
        i_val += (diff + pre_diff) * period *0.5f;

        //積分値が制限を超えてないかチェック
        if(inte_limit_flag) {
            i_val = i_val>inte_limit_max? inte_limit_max: i_val;
            i_val = i_val<inte_limit_min? inte_limit_min: i_val;
        }

        //操作量を計算
        float ctrl = kp * diff + kd * d_val + ki * i_val;

        //操作量が制限を超えてないかチェック
        if(ctrl_limit_flag) {
            ctrl = ctrl>ctrl_limit_max? ctrl_limit_max: ctrl;
            ctrl = ctrl<ctrl_limit_min? ctrl_limit_min: ctrl;
        }

        return ctrl;
    };

private:
    bool lpf_flag;//ローパスフィルタの有効化のフラグ
    bool ctrl_limit_flag;//操作量に制限をかけるかのフラグ
    bool inte_limit_flag;//積分値に制限をかけるかのフラグ

    float kp, ki, kd;//各ゲイン

    float period;//周期
    float target;//目標値

    float pre_diff;//前周期の制御量
    float i_val;//積分値

    float lpf_weight;//LPFの重み

    float ctrl_limit_max, ctrl_limit_min;//操作量の制限
    float inte_limit_max, inte_limit_min;//積分値の制限
};
#endif