/*
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   adi_sense_spi.h
 * @brief:  ADISENSE OS-dependent wrapper layer for SPI interface
 *-----------------------------------------------------------------------------
 */

#ifndef __ADI_SENSE_SPI_H__
#define __ADI_SENSE_SPI_H__

#include "inc/adi_sense_types.h"
#include "inc/adi_sense_platform.h"

/*! @ingroup ADISENSE_Host */

/*! @addtogroup ADISENSE_Spi ADISENSE Host SPI interface functions
 *  @{
 */

/*! A handle used in all API functions to identify the SPI interface context */
typedef void * ADI_SENSE_SPI_HANDLE;

#ifdef __cplusplus
extern "C"
{
#endif

/*!
 * @brief Open the SPI interface and allocate resources
 *
 * @param[in]  pConfig  Pointer to platform-specific SPI interface details
 * @param[out] phDevice Pointer to return a SPI interface context handle
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully
 *         - #ADI_SENSE_NO_MEM  Failed to allocate memory for interface context
 */
ADI_SENSE_RESULT adi_sense_SpiOpen(
    ADI_SENSE_PLATFORM_SPI_CONFIG * pConfig,
    ADI_SENSE_SPI_HANDLE          * phDevice);

/*!
 * @brief Close SPI interface and free resources
 *
 * @param[in] hDevice SPI interface context handle (@ref adi_sense_SpiOpen)
 */
void adi_sense_SpiClose(
    ADI_SENSE_SPI_HANDLE            hDevice);

/*!
 * @brief Transfer data to slave device
 *
 * @param[in] hDevice SPI interface context handle (@ref adi_sense_SpiOpen)
 * @param[in] pTxData Transmit data buffer, or NULL for read-only transfers
 * @param[in] pRxData Receive  data buffer, or NULL for write-only transfers
 * @param[in] nLength Number of bytes to transfer
 * @param[in] bCsHold Leave the chip-select asserted when the transfer is done
 *
 * @return Status
 *         - #ADI_SENSE_SUCCESS Call completed successfully
 *         - #ADI_SENSE_FAILURE Failed to complete SPI transfer
 */
ADI_SENSE_RESULT adi_sense_SpiTransfer(
    ADI_SENSE_SPI_HANDLE            hDevice,
    void                          * pTxData,
    void                          * pRxData,
    unsigned                        nLength,
    bool                            bCsHold);

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_SPI_H__ */
