
#include "mbed.h"
#include "Hotboards_keypad.h"
#include <ctype.h>

// Define the keymaps.  The blank spot (lower left) is the space character.
char alphaKeys[ 4 ][ 4 ] =
{
    { 'a' , 'd' , 'g' },
    { 'j' , 'm' , 'p' },
    { 's' , 'v' , 'y' },
    { ' ' , '.' , '#' }
};

char numberKeys[ 4 ][ 4 ] =
{
    { '1' , '2' , '3' },
    { '4' , '5' , '6' },
    { '7' , '8' , '9' },
    { ' ' , '0' , '#' }
};

bool alpha = false;   // Start with the numeric keypad.

// Defines the pins connected to the rows
DigitalInOut rowPins[ 4 ] = { PA_6 , PA_7 , PB_6 , PC_7 };
// Defines the pins connected to the cols
DigitalInOut colPins[ 4 ] = { PA_8 , PB_10 , PB_4 , PB_5 };

// Create two new keypads, one is a number pad and the other is a letter pad.
Keypad numpad( makeKeymap( numberKeys ) , rowPins , colPins , 4 , 4 );
Keypad ltrpad( makeKeymap( alphaKeys ) , rowPins , colPins , 4 , 4 ); 

int startTime;
// For this example we will use the Nucleo LED1 on pin PA_5
DigitalOut led1( LED1 );

// Configures the serial port
Serial pc( USBTX , USBRX );

// Configures a timer
Timer t;

char key;
static char virtKey = NO_KEY;   // Stores the last virtual key press. (Alpha keys only)
static char physKey = NO_KEY;   // Stores the last physical key press. (Alpha keys only)
static char buildStr[ 12 ];
static uint8_t buildCount;
static uint8_t pressCount;
static uint8_t kpadState;

// Take care of some special events.

void swOnState( char key )
{
    switch( kpadState )
    {
        case PRESSED:
            if( isalpha( key ) )        // This is a letter key so we're using the letter keymap.
            {
                if( physKey != key )    // New key so start with the first of 3 characters.
                {
                    pressCount = 0;
                    virtKey = key;
                    physKey = key;
                }
                else                    // Pressed the same key again...
                {
                    virtKey ++;         // so select the next character on that key.
                    pressCount ++;      // Tracks how many times we press the same key.
                }
                if( pressCount > 2 )    // Last character reached so cycle back to start.
                {
                    pressCount = 0;
                    virtKey = key;
                }
                pc.printf( "%c" , virtKey );    // Used for testing.
                if( isdigit( key ) || key == ' ' || key == '.' )
                {
                    pc.printf( "%c" , key );
                }
                if( key == '#' )
                {
                    pc.printf( "\n\r" );
                }
                break;
            }
        case HOLD:
            if( key == '#' )            // Toggle between keymaps.
            {
                if( alpha == true )     // We are currently using a keymap with letters
                {
                    alpha = false;      // Now we want a keymap with numbers.
                    led1 = 0;
                }
                else                    // We are currently using a keymap with numbers
                {
                    alpha = true;       // Now we want a keymap with letters.
                }
            }
            else                        // Some key other than '#' was pressed.
            {
                buildStr[ buildCount ++ ] = ( isalpha( key ) ) ? virtKey : key;
                buildStr[ buildCount ] = '\0';
                pc.printf( "\n\r" );
                pc.printf( buildStr );
            }
            break;
        case RELEASED:
            if( buildCount >= sizeof( buildStr ) )  // Our string is full. Start fresh.
            {
                buildCount = 0;
            }
            break;
    }
}

void keypadEvent_ltr( KeypadEvent key )
{
    // in here when in alpha mode.
    kpadState = ltrpad.getState( );
    swOnState( key );
}

void keypadEvent_num( KeypadEvent key )
{
    // in here when using number keypad
    kpadState = numpad.getState( );
    swOnState( key );
}

int main()
{
    // Starts the timer
    t.start( );
    led1 = 0;   // Turns the LED off.
    ltrpad.begin( makeKeymap( alphaKeys ) );
    numpad.begin( makeKeymap( numberKeys ) );
    ltrpad.addEventListener( keypadEvent_ltr ); // Add an event listener.
    ltrpad.setHoldTime( 500 );                  // Default is 1000mS
    numpad.addEventListener( keypadEvent_num ); // Add an event listener.
    numpad.setHoldTime( 500 );                  // Default is 1000mS
    
    while(1)
    {
        if( alpha )
        {
            key = ltrpad.getKey( );
        }
        else
        {
            key = numpad.getKey( );
        }
        if( alpha && t.read_ms( ) - startTime > 100 )
        {
            // Flash the LED if we are using the letter keymap.
            led1 = !led1;
            startTime = t.read_ms( );
        }
    }
}

