/*
 * Copyright (C)2010-2012 Henning Karlsen. All right reserved.
 * Copyright (C)2012 Todor Todorov.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to:
 *
 * Free Software Foundation, Inc.
 * 51 Franklin St, 5th Floor, Boston, MA 02110-1301, USA
 *
 *********************************************************************/
#include "ssd1963.h"
#include "helpers.h"

SSD1963_LCD::SSD1963_LCD( PinName CS, PinName RESET, PinName RS, PinName WR, BusOut* DATA_PORT, PinName BL, PinName RD, backlight_t blType, float defaultBackLightLevel )
    : LCD( 272, 480, CS, RS, RESET, BL, blType, defaultBackLightLevel ), _lcd_pin_wr( WR )
{
    _lcd_port = DATA_PORT;
    if ( RD != NC ) _lcd_pin_rd = new DigitalOut( RD );
    else _lcd_pin_rd = 0;
}

void SSD1963_LCD::Initialize( orientation_t orientation, colordepth_t colors )
{
    _orientation = orientation;
    _colorDepth = colors;
    
    _lcd_pin_reset = HIGH;
    wait_ms( 5 );
    _lcd_pin_reset = LOW;
    wait_ms( 15 );
    _lcd_pin_reset = HIGH;
    _lcd_pin_cs = HIGH;
    if ( _lcd_pin_bl != 0 )
        *_lcd_pin_bl = HIGH;
    else if ( _bl_pwm != 0 )
        *_bl_pwm = _bl_pwm_default;
    if ( _lcd_pin_rd != 0 )
        *_lcd_pin_rd = HIGH;
    _lcd_pin_wr = HIGH;
    wait_ms( 15 );
    
    Activate();
    // Set PLL MN -------------------------------------------------------------
    // @Parameters : 3
    WriteCmd(0xE2);
    WriteData(0x23);             // N[7:0] : Multiplier (N) of PLL. (POR = 00101101) b00100011
    WriteData(0x02);             // M[3:0] : Divider (M) of PLL. (POR = 0011)
    WriteData(0x04);             // C[2] : Effectuate MN value (POR = 100) - Effectuate the multiplier and divider value
    //WriteData(0x54);
    
    // Set PLL
    // @Parameters : 1
    WriteCmd(0xE0);
    WriteData(0x01);             // Use reference clock as system clock & Enable PLL
    wait_ms(1);                  // Wait 100us to let the PLL stable
    WriteCmd(0xE0);             // Set PLL
    WriteData(0x03);             // Use PLL output as system clock & Enable PLL
    wait_ms(1); 

    // Software Reset ---------------------------------------------------------
    WriteCmd(0x01);
    wait_ms(1);   

    // Set LSHIFT Frequency ---------------------------------------------------
    // @Parameters : 3
    WriteCmd(0xE6);             // Set LSHIFT Frequency
    WriteData(0x03);             // LCDC_FPR[19:16] : The highest 4 bits for the pixel clock frequency settings
    WriteData(0xFF);             // LCDC_FPR[15:8]  : The higher byte for the pixel clock frequency settings
    WriteData(0xFF);             // LCDC_FPR[7:0]   : The low byte for the pixel clock frequency settings

    // Set LCD Mode
    // @Parameters : 7
    WriteCmd(0xB0);
    /*
    WriteData(0x00);             // A[5..0] TFT
    //WriteData(0x10);
    WriteData(0x00);             // B[7..5]   : Hsync+Vsync +DE mode & TFT mode
    //WriteData(0x80);
    WriteData((HDP>>8)&0xFF);    // HPS[10:8] : Set the horizontal panel size (POR = 010)
    WriteData(HDP&0xFF);         // HPS[7:0]  : Set the horizontal panel size (POR = 01111111)
    WriteData((VDP>>8)&0xFF);    // VPS[10:8] : Set the vertical panel size (POR = 001)
    WriteData(VDP&0xFF);         // VPS[7:0]  : Set the vertical panel size (POR = 11011111)
    WriteData(0x00);             // G[5..0]   : Even line RGB sequence & Odd line RGB sequence
    */
    //WriteData(0x10);                // set 18-bit for 7" panel TY700TFT800480
    WriteData(0x20);                // set 18-bit for 7" panel TY700TFT800480
    //WriteData(0x80);                // set TTL mode
    WriteData(0x00);                // set TTL mode
    WriteData((DISP_HOR_RESOLUTION-1)>>8); //Set panel size
    WriteData(DISP_HOR_RESOLUTION-1);
    WriteData((DISP_VER_RESOLUTION-1)>>8);
    WriteData(DISP_VER_RESOLUTION-1);
    WriteData(0x00);
    
    
    // Set Horizontal Period --------------------------------------------------
    // @Parameters : 8
    WriteCmd(0xB4);
    /*
    WriteData((HT>>8)&0xFF);     // HT[10:8]   : High byte of horizontal total period (display + non-display) in pixel clock
    WriteData(HT&0xFF);          // HT[7:0]    : Low byte of the horizontal total period (display + non-display) in pixel clock
    WriteData((HPS>>8)&0xFF);    // HPS[10:8]  : High byte of the non-display period between the start of the horizontal sync (LLINE) signal
    WriteData(HPS&0xFF);         // HPS[7:0]   : Low byte of the non-display period between the start of the horizontal sync (LLINE) signal
    WriteData(HPW&0xFF);         // HPW[6:0]   : Set the horizontal sync pulse width (LLINE) in pixel clock
    WriteData((LPS>>8)&0xFF);    // LPS[10:8]  : Set the horizontal sync pulse (LLINE) start location in pixel clock
    WriteData(LPS&0xFF);         // LPS[7:0]   : Set the horizontal sync pulse width (LLINE) in start.
    WriteData(0x00);             // LPSPP[1:0] : Set the horizontal sync pulse subpixel start position
    */
    #define HT (DISP_HOR_RESOLUTION+DISP_HOR_PULSE_WIDTH+DISP_HOR_BACK_PORCH+DISP_HOR_FRONT_PORCH)
    WriteData((HT-1)>>8);   
    WriteData(HT-1);
    #define HPS (DISP_HOR_PULSE_WIDTH+DISP_HOR_BACK_PORCH)
    WriteData((HPS-1)>>8);
    WriteData(HPS-1);
    WriteData(DISP_HOR_PULSE_WIDTH-1);
    WriteData(0x00);
    WriteData(0x00);
    WriteData(0x00);
    
    // Set Vertical Period ----------------------------------------------------
    // @Parameters : 7
    WriteCmd(0xB6);
    /*
    WriteData((VT>>8)&0xFF);     // VT[10:8]  : High byte of the vertical total (display + non-display) period in lines
    WriteData(VT&0xFF);          // VT[7:0]   : Low byte of the vertical total (display + non-display) period in lines
    WriteData((VPS>>8)&0xFF);    // VPS[10:8] : High byte the non-display period in lines between the start of the frame and the first display data in line
    WriteData(VPS&0xFF);         // VPS[7:0]  : The non-display period in lines between the start of the frame and the first display data in line
    WriteData(VPW&0xFF);         // VPW[6:0]  : Set the vertical sync pulse width (LFRAME) in lines
    WriteData((FPS>>8)&0xFF);    // FPS[10:8] : High byte of the vertical sync pulse (LFRAME) start location in lines
    WriteData(FPS&0xFF);         // FPS[7:0]  : Low byte of the vertical sync pulse (LFRAME) start location in lines
    */
    #define VT (DISP_VER_PULSE_WIDTH+DISP_VER_BACK_PORCH+DISP_VER_FRONT_PORCH+DISP_VER_RESOLUTION)
    WriteData((VT-1)>>8);
    WriteData(VT-1);
    #define VSP (DISP_VER_PULSE_WIDTH+DISP_VER_BACK_PORCH)
    WriteData((VSP-1)>>8);
    WriteData(VSP-1);
    WriteData(DISP_VER_PULSE_WIDTH-1);
    WriteData(0x00);
    WriteData(0x00);
    
    // Set GPIO Value ---------------------------------------------------------
    // @Parameters : 1
    WriteCmd(0xBA);
    WriteData(0x05);             // A[3..0] : GPIO[2:0] Output 1
    
    // Set GPIO Configuration
    // @Parameters : 2
    WriteCmd(0xB8);
    WriteData(0x07);             // A[7..0] : GPIO3 = Input, GPIO[2:0] = Output
    WriteData(0x01);             // B[0] : GPIO0 Normal
    
        //Set pixel format, i.e. the bpp
    WriteCmd(0x3A);
    WriteData(0x55);                // set 16bpp

    // Set Address Mode -------------------------------------------------------
    // @Parameters : 1
    //WriteCmd(0x36);             // Set Rotation
    //WriteData(0x00);             // A[7..0] : Set the read order from host processor to frame buffer by A[7:5] and A[3] and 
                                                       // from frame buffer to the display panel by A[2:0] and A[4].
                                                       // A[7] : Page address order
    // Set Pixel Data Interface -----------------------------------------------
    // @Parameters : 1
    WriteCmd(0xF0);             // A[2:0] : Pixel Data Interface Format
    WriteData(0x03);             // 16-bit (565 format)
    wait_ms(1);
    
    // enter_partial_mode
    //WriteCmd(0x12);             // Part of the display area is used for image display
    // set_display_on
    WriteCmd(0x29);             // Show the image on the display device

    Deactivate();
}

void SSD1963_LCD::Sleep( void )
{
    WriteCmdData( 0x10, 0x0001 ); // sleep mode: 0 = exit, 1 = enter
    LCD::Sleep();
}

void SSD1963_LCD::WakeUp( void )
{
    WriteCmdData( 0x10, 0x0000 ); // sleep mode: 0 = exit, 1 = enter
    LCD::WakeUp();
}

void SSD1963_LCD::WriteCmd( unsigned short cmd )
{
    _lcd_pin_rs = LOW;
    _lcd_port->write( cmd );
    _lcd_pin_wr = LOW;
    wait_us(1);
    _lcd_pin_wr = HIGH;
    //pulseLow( _lcd_pin_wr );
}

void SSD1963_LCD::WriteData( unsigned short data )
{
    _lcd_pin_rs = HIGH;
    _lcd_port->write( data );
    _lcd_pin_wr = LOW;
    wait_us(1);
    _lcd_pin_wr = HIGH;
    
    //pulseLow( _lcd_pin_wr );
}

void SSD1963_LCD::SetXY( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2 )
{
    //if ( _orientation == PORTRAIT || _orientation == PORTRAIT_REV )
    //{
        WriteCmd( 0x2A);
        WriteData( x1>>8 );
        WriteData( x1 );
        WriteData( x2>>8 );
        WriteData( x2 );
        WriteCmd( 0x2B);
        WriteData( y1>>8 );
        WriteData( y1 );
        WriteData( y2>>8 );
        WriteData( y2 );
        WriteCmd( 0x2B);
    /*}
    else
    {
        WriteCmd( 0x2A);
        WriteData( y1>>8 );
        WriteData( y1 );
        WriteData( y2>>8 );
        WriteData( y2 );
        WriteCmd( 0x2B);
        WriteData( x1>>8 );
        WriteData( x1 );
        WriteData( x2>>8 );
        WriteData( x2 );
        WriteCmd( 0x2B);
    }*/
    WriteCmd( 0x2C );
}

void SSD1963_LCD::SetPixelColor( unsigned int color, colordepth_t mode )
{
    unsigned char r, g, b;
    unsigned short clr;
    if ( _colorDepth == RGB16 )
    {
        switch ( mode )
        {
            case RGB16:
                WriteData( color & 0xFFFF );
                break;
            case RGB18:
                r = ( color >> 10 ) & 0xF8;
                g = ( color >> 4 ) & 0xFC;
                b = ( color >> 1 ) & 0x1F;
                clr = ( ( r | ( g >> 5 ) ) << 8 ) | ( ( g << 3 ) | b );
                WriteData( clr );
                break;
            case RGB24:
                r = ( color >> 16 ) & 0xF8;
                g = ( color >> 8 ) & 0xFC;
                b = color & 0xF8;
                clr = ( ( r | ( g >> 5 ) ) << 8 ) | ( ( g << 3 ) | ( b >> 3 ) );
                WriteData( clr );
                break;
        }
    }
    else if ( _colorDepth == RGB18 )
    {
        switch ( mode )
        {
            case RGB16:
                r = ( ( color >> 8 ) & 0xF8 ) | ( ( color & 0x8000 ) >> 13 );
                g = ( color >> 3 ) & 0xFC;
                b = ( ( color << 3 ) & 0xFC ) | ( ( color >> 3 ) & 0x01 );
                WriteData( ( r << 8 ) | g );
                WriteData( b );
                break;
            case RGB18:
                b = ( color << 2 ) & 0xFC;
                g = ( color >> 4 ) & 0xFC;
                r = ( color >> 10 ) & 0xFC;
                WriteData( ( r << 8 ) | g );
                WriteData( b );
                break;
            case RGB24:
                r = ( color >> 16 ) & 0xFC;
                g = ( color >> 8 ) & 0xFC;
                b = color & 0xFC;
                WriteData( ( r << 8 ) | g );
                WriteData( b );
                break;
        }
    }
}
