/************************************************************************
 * Simple indoor weather station with TFT display
 *
 * Sensors: AM2302 aka DHT22 hygrometer thermometer
 *      or  SHT10/SHT11/SHT15 hygrometer thermometer
 *          BMP180 barometer atmospheric pressure
 *
 * Real time clock: DS3231
 * Storage:         24c32 (on rtc board)
 *
 * Board: ST NUCLEO-L152RE
 *        cheap 2.2inch 320x240 TFT with SPI-Interface and SDCard socket.
 *
 * Data stored every 10 minutes into file and 8K byte EEPROM
 *
 * EEPROM storage:
 * 576 values of temp, hygro and barometer (about 4 days)
 * Format: day, hour, min, temp, humidity, preasure
 *         pressure will cover 940hpa to 1060hpa
 *
 * Similar program exists for a E-Paper display and Seeduino Arch Pro.
 * will release this code later.
 *
 * Changes:
 * June, 1st :  Version 2.0
 *              Changed to landscape view and inverted display
 *              add support for SHT11 sensor. 
 *              SHT10 and SHT15 should also work.
 *              Using TTP224 capacity toch switch with 4 sensors
 *              Touch 1 = show statistics graph
 *              Touch 2 = show normal display
 *              Touch 3 = (not used yet)
 *              Touch 4 = toggle diplay led on/off
 * May, 18.: Now guessing weather. First splash screen disabled.
 *           Changed the reading of temp sensor. If after 5 tries 
 *           no luck, take old values. The AM2302 reading is not
 *           very reliable.
 ************************************************************************/

//#define NO_DMA

#include "stdio.h"
#include "mbed.h"
#include "string"

// TFT and SDCard
#include "SDFileSystem.h"
#include "SPI_TFT_ILI9341.h"

// RTC, EEPROM sensors
#include "DS3231.h"
#include "_24LCXXX.h"
#include "BMP180.h"
#include "RHT03.h" //AM2303
#include "sht15.hpp"

// some fonts
#include "Courier9x15.h"
#include "Arial12x12.h"
#include "Arial28x28.h"
#include "Times_New_Roman28x27.h"

// my functions
void drawClock(void);
void drawAnalogTime(void);
void drawDigitalTime(void);
void drawTempAM2302(void);
void drawTempSHT(void);
void drawPressure(void);
void SetTime(void);
void UpdateDisplay(void);
void UpdateDisplayNoStore(void);
void switchBackLight(void);
void storeData(void);
void showGraph(void);
int map (int ,int , int , int , int);

char store_temp, store_humidity, store_pressure;

extern unsigned char p1[];  // the mbed logo

// analog clock size and locate
#define CLOCK_RADIUS 64           // radius of clock face
#define CLOCK_CENTER_X 82         // If you adjust the radius, you'll probably want to adjust this
#define CLOCK_CENTER_Y 150         // If you adjust the radius, you'll probably want to adjust this

// Maximum of stored value into eeprom
#define max_datastorage 576       // 576 * 6 = 3456
#define COUNTERADDRESS 4000       // The counter for ring buffer will be stored here 

#define RAIN    1000
#define CHANGE  1010
#define FAIR    1020
#define SUNNY   1030 // hPa

#define BGCOLOR    0x0000
#define TEXTCOLOR  0xFFFF

// the TFT is connected to SPI
SPI_TFT_ILI9341 TFT(PA_7, PA_6, PA_5, PB_6, PA_9, PC_7,"TFT"); // NUCLEO mosi, miso, sclk, cs, reset, dc

// SDCard on SPI2 at morpho connectors
SDFileSystem sd(PB_15, PB_14, PB_13, PB_12, "sd"); // MOSI, MISO, SCLK, SSEL

I2C i2c(PB_9, PB_8);            // sda, scl

BMP180 bmp(PB_9, PB_8);
DS3231 rtc(PB_9,PB_8);
_24LCXXX eeprom(&i2c, 0x57);
SHTx::SHT15 sht11(PB_11, PB_10);// SDA,SCL extra I2C port needed for sht1x sensor
RHT03 humtemp(PA_10);           //Initalise the RHT03 (change pin number to the pin its connected to)

// set interrupts
InterruptIn RTCInt(PC_4);       // from rtc module
InterruptIn ShowGraphInt(PC_2); // from touch pad [1]
InterruptIn UpDispInt(PC_3);    // from touch pad [2]
//InterruptIn   (PC_10);        // not yet used   [3]
InterruptIn BackLight(PC_5);    // from touch pad [4]

DigitalOut TFTBL(PB_3);         // backlit control

int main()
{
    char Control;
    int err;

    // Test if SD-Card works
    //FILE *fp = fopen("/sd/mbed.txt", "a");
    //fprintf(fp, "Hello World!\n");
    //fclose(fp);
    
    TFTBL = 1;  // Switch back light on

    TFT.set_orientation(1); // Oriantation landscape
    TFT.claim(stdout);      // send stdout to the TFT display
    TFT.claim(stderr);      // send stderr to the TFT display
    TFT.background(BGCOLOR);    // set background color
    TFT.foreground(TEXTCOLOR);    // set chars color
    TFT.set_font((unsigned char*) Arial12x12);
    TFT.cls();                // clear the screen

    /*
    err = TFT.BMP_16(9,12,"/sd/Carina.bmp");  // load test.bmp from external SD-card
    TFT.locate(10,120);
    if (err != 1) TFT.printf(" - Err: %d",err);
    wait(2);
    TFT.cls();                // clear the screen
    */
    
    sht11.update();
    
    
    TFT.locate(0,0);
    printf("  Hello mbed.org ");
    TFT.locate(100,0);
    printf("Temp: %2.2f  Hum: %2.2f",sht11.getTemperature(),sht11.getHumidity());
    //TFT.Bitmap(20,20,172,55,p1);
    err = TFT.BMP_16(2,20,"/sd/sun_b.bmp");  // load bmp from external SD-card
    if (err != 1) TFT.printf(" - Err: %d",err);
    err = TFT.BMP_16(100,20,"/sd/suncloud_b.bmp");  // load bmp from external SD-card
    if (err != 1) TFT.printf(" - Err: %d",err);
    err = TFT.BMP_16(200,20,"/sd/cloud_b.bmp");  // load bmp from external SD-card
    if (err != 1) TFT.printf(" - Err: %d",err);
    err = TFT.BMP_16(2,120,"/sd/rain_b.bmp");  // load bmp from external SD-card
    if (err != 1) TFT.printf(" - Err: %d",err);
    err = TFT.BMP_16(100,120,"/sd/thunder_b.bmp");  // load bmp from external SD-card
    if (err != 1) TFT.printf(" - Err: %d",err);


    // set the DS3231 for alarm every minute.
    rtc.writeRegister(DS3231_Alarm2_Minutes,0x80);
    rtc.writeRegister(DS3231_Alarm2_Hours,0x80);
    rtc.writeRegister(DS3231_Alarm_2_Day_Date,0x80);

    Control = rtc.readRegister(DS3231_Control);
    Control = Control|DS3231_bit_A2IE;
    Control = Control|DS3231_bit_INTCN;
    rtc.writeRegister(DS3231_Control,Control);

    Control=rtc.readRegister(DS3231_Control_Status)&0xFC;
    rtc.writeRegister(DS3231_Control_Status,Control);


    wait(2);        // wait a second

    UpdateDisplay();

    ShowGraphInt.rise(&showGraph);  // interrupt for showing value history diagram. called by a button or capacitive touch switch

    BackLight.rise(&switchBackLight);  // interrupt for showing value history diagram. called by a button or capacitive touch switch
    
    UpDispInt.rise(&UpdateDisplayNoStore);

    RTCInt.fall(&UpdateDisplay);     // interrupt every minute called by rtc alarm for display update

    while(1) {
        wait(10.0);
    }
}

void switchBackLight(void)
{
    TFTBL = !TFTBL;
    }

void UpdateDisplay(void)
{

    //led2 = 0;

    TFT.cls();

    TFT.fillrect(317,237,319,239,Red); // Just for status: during update there is a small red dot at the bottom right corner

    drawClock();
    drawAnalogTime();
    drawDigitalTime();

    drawPressure();

    drawTempSHT();

    storeData();

    // Reset clock alarm
    char Control=rtc.readRegister(DS3231_Control_Status)& 0xFC;
    rtc.writeRegister(DS3231_Control_Status,Control);

    //led2 = 1;

}

void UpdateDisplayNoStore(void)
{   // Data should only stored on rtc interrupt

    TFT.cls();

    TFT.fillrect(317,237,319,239,Red); // Just for status: during update there is a small red dot at the bottom right corner

    drawClock();
    drawAnalogTime();
    drawDigitalTime();

    drawPressure();

    drawTempAM2302();

}


// store every ten minute values into file and eeprom
void storeData(void)
{
    int date, month, year, hour, minute, second, dayOfWeek;
    unsigned int store_address;
    char b[6];

    rtc.readDateTime(&dayOfWeek,&date,&month,&year,&hour,&minute,&second);

    // Save only every 10 minutes
    if ( minute%10 == 0) {

        /* Store into EEPROM */
        eeprom.nbyte_read( COUNTERADDRESS, &store_address, sizeof(int) );

        if ( store_address > (max_datastorage * 6)) {
            store_address = 0;
        }

        b[0] = date;
        b[1] = hour;
        b[2] = minute;
        b[3] = store_temp;
        b[4] = store_humidity;
        b[5] = store_pressure;

        eeprom.nbyte_write( store_address, &b, 6);

        store_address += 6;

        eeprom.nbyte_write(COUNTERADDRESS,&store_address,sizeof(int));

        // Store into file
        FILE *fp = fopen("/sd/stored_data.txt", "a");
        fprintf(fp, "%02i %02i %02i:%02i %02i %cC %02i %% %04i hPa\n",month, date, hour, minute, store_temp, 176, store_humidity, (store_pressure+1880) * 50 / 100);
        fclose(fp);

    }

}

// show grap of stored data (at the moment landscape oriented. not yet changed to potrait display)
void showGraph(void)
{
    unsigned int store_address,i;
    char b[6];
    char temp_arr[250], temp_min, temp_max;
    char humidity_arr[250], hum_min, hum_max;
    int pressure_arr[250], press_min, press_max;

    eeprom.nbyte_read( COUNTERADDRESS, &store_address, sizeof(int) );

    TFTBL = 1;  // Switch back light on

    TFT.set_font((unsigned char*) Courier9x15);

    //TFT.set_orientation(1);
    TFT.cls();
    TFT.line(68,8,68,50,TEXTCOLOR);
    TFT.line(66,48,319,48,TEXTCOLOR);
    TFT.line(68,80,68,122,TEXTCOLOR);
    TFT.line(66,120,319,120,TEXTCOLOR);
    TFT.line(68,152,68,204,TEXTCOLOR);
    TFT.line(66,202,319,202,TEXTCOLOR);

    // store_address is allways one step ahead of last stored values
    if ( store_address == 0)
        store_address = max_datastorage * 6;
    else
        store_address -= 6;

    // read from eeprom, store into array, find min and max values
    temp_min = 100;
    hum_min = 99;
    press_min = 1150;
    temp_max = 0;
    hum_max = 0;
    press_max = 0;

    // Read from EEPROM storage into arrays
    for (i=0; i<250; i++) {
        eeprom.nbyte_read( store_address, &b, 6);

        // draw every 2 hour an line
        if ( b[2] == 0 && (b[1]%4 == 0 || b[1] == 0) ) {
            TFT.line(319-i,46,319-i,50,TEXTCOLOR);
            TFT.line(319-i,118,319-i,122,TEXTCOLOR);
            TFT.line(319-i,200,319-i,204,TEXTCOLOR);
            if ( i > 15 )                     // do not write at the right border
            {
                TFT.locate(310-i,208);
                printf("%2i",b[1]);
            }
        }

        // store temp into array and check min max
        temp_arr[i] = b[3];
        if ( b[3] < temp_min )
            temp_min = b[3];
        if ( b[3] > temp_max )
            temp_max = b[3];

        // store humidity into array and check min max
        humidity_arr[i] = b[4];
        if ( b[4] < hum_min )
            hum_min = b[4];
        if ( b[4] > hum_max )
            hum_max = b[4];

        // store pressure into array and check min max
        pressure_arr[i] = (b[5]+1880) * 50 / 100; // calculate pressure to Pascal and hektoPascal
        if ( pressure_arr[i] < 990 )
            pressure_arr[i] = 1000;

        if ( pressure_arr[i] < press_min )
            press_min = pressure_arr[i];
        if ( pressure_arr[i] > press_max )
            press_max = pressure_arr[i];

        if ( store_address == 0)
            store_address = max_datastorage * 6;
        else
            store_address -= 6;
    }

    // rounding min and max to 5
    temp_min -= 1;
    temp_min = temp_min - (temp_min%5);
    temp_max += 1;
    temp_max = temp_max + (5 - (temp_max%5));
    TFT.locate(0,2);
    printf("%i%cC",temp_max,127);
    TFT.locate(0,40);
    printf("%i%cC",temp_min,127);

    hum_min -= 1;
    hum_min = hum_min - (hum_min%5);
    hum_max += 1;
    hum_max = hum_max + (5 - (hum_max%5));
    TFT.locate(0,74);
    printf("%i%%",hum_max);
    TFT.locate(0,112);
    printf("%i%%",hum_min);

    press_min -= 1;
    press_min = press_min - (press_min%5);
    press_max += 1;
    press_max = press_max + (5 - (press_max%5));
    TFT.locate(0,146);
    printf("%ihPa",press_max);
    TFT.locate(0,184);
    printf("%ihPa",press_min);

    // drawing lines
    for (i=0; i<250; i++) {
        // temp line
        TFT.pixel(319-i,map(temp_arr[i],temp_min,temp_max,48,8),Red);
        // humidity line
        TFT.pixel(319-i,map(humidity_arr[i],hum_min,hum_max,120,80),Blue);
        // pressure line
        TFT.pixel(319-i,map(pressure_arr[i],press_min,press_max,202,152),Green);

    }

}

// draw temp and humidity
void drawTempAM2302(void)
{
    int done=0, i;
    float temp,hum;

    for (i=0; i<5;i++) {      //try to read 5 times (10secs)
        wait(2.2);            //Needed to make sure the sensor has time to initalise and so its not polled too quickly
        if(humtemp.readData() == RHT_ERROR_NONE) { //Request data from the RHT03
            done=1;
            i=5;
        }
    }

    if(done) { // only if read was successful
        temp = humtemp.getTemperatureC(); //Gets the current temperature in centigrade
        store_temp = temp;
        hum = humtemp.getHumidity(); //Gets the current humidity in percentage
        store_humidity = hum;
        TFT.fillrect(317,237,319,239,BGCOLOR);
    } else { // read was not successful, take old values
        temp = store_temp;
        hum = store_humidity;
        TFT.fillrect(317,237,319,239,0x7000);
    }

    //TFT.set_font((unsigned char*) Times_New_Roman28x27);  // select the font
    TFT.set_font((unsigned char*) Arial28x28);  // select the font
    TFT.locate(232,0);
    printf("%2.0f%cC",temp,127);
    TFT.locate(240,40);
    printf("%2.0f%%",hum);

}

void drawTempSHT(void)
{
    float temp,hum;

    sht11.update();

    temp = sht11.getTemperature(); //Gets the current temperature
    store_temp = temp;
    hum = sht11.getHumidity(); //Gets the current humidity in percentage
    store_humidity = hum;
    TFT.fillrect(317,237,319,239,BGCOLOR);

    //TFT.set_font((unsigned char*) Times_New_Roman28x27);  // select the font
    TFT.set_font((unsigned char*) Arial28x28);  // select the font
    TFT.locate(232,0);
    printf("%2.0f%cC",temp,127);
    TFT.locate(240,40);
    printf("%2.0f%%",hum);

}

// read and draw air pressure. later calculate symbol for wheather guess regarding pressure change.
void drawPressure(void)
{
    long Pressure, BMP180_Temperature;

    bmp.readTP(&BMP180_Temperature,&Pressure,OVERSAMPLING_STANDARD);

    store_pressure = char((Pressure / 50) - 1880);

    Pressure /= 100; // convert to hPa

    //TFT.set_font((unsigned char*) Times_New_Roman28x27);  // select the font
    TFT.set_font((unsigned char*) Arial28x28);  // select the font
    TFT.locate(180,200);
    printf("%4dhPa",Pressure);
    TFT.set_font((unsigned char*) Courier9x15);
    TFT.locate(220,100);
    printf("%i,%1i%cC",BMP180_Temperature/10,BMP180_Temperature%10,127);

    // Draw some weather icons for test
    int err ;
    if ( Pressure < RAIN ) {
        err = TFT.BMP_16(192,92,"/sd/rain_b.bmp");  // load test.bmp from external SD-card
        if (err != 1) TFT.printf(" - Err: %d",err);
    } else if ( Pressure < CHANGE ) {
        err = TFT.BMP_16(192,92,"/sd/cloud_b.bmp");  // load test.bmp from external SD-card
        if (err != 1) TFT.printf(" - Err: %d",err);
    } else if ( Pressure < FAIR ) {
        err = TFT.BMP_16(192,92,"/sd/suncloud_b.bmp");  // load test.bmp from external SD-card
        if (err != 1) TFT.printf(" - Err: %d",err);
    } else {
        err = TFT.BMP_16(192,92,"/sd/sun_b.bmp");  // load test.bmp from external SD-card
        if (err != 1) TFT.printf(" - Err: %d",err);
    }

}

// draw digital date and time.
void drawDigitalTime()
{
    //SetTime(); // Call only once to set date and time on DS3231

    int date, month, year, hour, minute, second, dayOfWeek;

    rtc.readDateTime(&dayOfWeek,&date,&month,&year,&hour,&minute,&second);

    //TFT.set_font((unsigned char*) Times_New_Roman28x27);  // select the font
    TFT.set_font((unsigned char*) Arial28x28);  // select the font

    if (date > 0) {
        TFT.locate(0,0);
        printf("%02i.%02i.%i",date,month,year);
        TFT.locate(44,40);
        printf("%02i:%02d",hour,minute);
    } else {
        TFT.locate(0,0);
        printf("Error read RTC");
    }

}

// guess what ;-)
// procedure to set time. mosty only used once for rtc
// rtc module wil keep time with battery
void SetTime()
{
    // DS3231
    // hour,minute,second
    rtc.setTime(23,39,0);
    // dayofweek,day,month,year
    rtc.setDate(3,10,5,2014);

}

// draw hands into analog clock
void drawAnalogTime()
{
    int x1, x2, y1, y2, x3, y3, m, h, midHours, second;

    rtc.readTime(&h,&m,&second);

    // Draw minute hand
    midHours = m/12;
    m -= 30;
    if ( m < 0 )
        m += 60;

    m = map(m,0,60,360,0);

    // Center for single line hand
    // double line hand
    x1 = CLOCK_CENTER_X + 3 * sin(3.14 * ((double) (m - 90))/180);
    y1 = CLOCK_CENTER_Y + 3 * cos(3.14 * ((double) (m - 90))/180);
    x3 = CLOCK_CENTER_X + 3 * sin(3.14 * ((double) (m + 90))/180);
    y3 = CLOCK_CENTER_Y + 3 * cos(3.14 * ((double) (m + 90))/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 20) * sin(3.14 * ((double) m)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 20) * cos(3.14 * ((double) m)/180);
    TFT.line(x1,y1,x2,y2,Red);
    TFT.line(x3,y3,x2,y2,Red);
    TFT.line(x1,y1,x3,y3,Red);

    // Draw hour hand

    // Calc from 24 to 12 hour
    if ( h > 11 )
        h -= 12;

    // Rotate 180°. Otherwise clock will be printed upside down
    h = h - 6;
    if ( h < 0 )
        h += 12;

    // minute correction for hour hand
    h *= 5;
    h += midHours;

    h = map(h,0,60,360,0);

    x1 = CLOCK_CENTER_X + 3 * sin(3.14 * ((double) (h - 90))/180);
    y1 = CLOCK_CENTER_Y + 3 * cos(3.14 * ((double) (h - 90))/180);
    x3 = CLOCK_CENTER_X + 3 * sin(3.14 * ((double) (h + 90))/180);
    y3 = CLOCK_CENTER_Y + 3 * cos(3.14 * ((double) (h + 90))/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 32) * sin(3.14 * ((double) h)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 32) * cos(3.14 * ((double) h)/180);
    TFT.line(x1,y1,x2,y2,Green);
    TFT.line(x3,y3,x2,y2,Green);
    TFT.line(x1,y1,x3,y3,Green);

}


// Draw analog clock w/o hands
void drawClock(void)
{
    int x1, x2, y1, y2;

    TFT.background(BGCOLOR);    // set background to black
    TFT.foreground(TEXTCOLOR);    // set chars to white

    TFT.fillcircle(CLOCK_CENTER_X,CLOCK_CENTER_Y,CLOCK_RADIUS,TEXTCOLOR);
    TFT.fillcircle(CLOCK_CENTER_X,CLOCK_CENTER_Y,CLOCK_RADIUS-4,BGCOLOR);

    /*
        TFT.set_font((unsigned char*) Arial12x12);
        TFT.locate( CLOCK_CENTER_X - 8, CLOCK_CENTER_Y  - CLOCK_RADIUS + 6);
        printf("12");
        TFT.locate( CLOCK_CENTER_X - CLOCK_RADIUS + 6, CLOCK_CENTER_Y - 6);
        printf("9");
        TFT.locate( CLOCK_CENTER_X + CLOCK_RADIUS - 12, CLOCK_CENTER_Y - 6);
        printf("3");
        TFT.locate( CLOCK_CENTER_X - 2, CLOCK_CENTER_Y + CLOCK_RADIUS - 15);
        printf("6");
    */

    x1 = CLOCK_CENTER_X - 1;
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 16);
    x2 = CLOCK_CENTER_X + 1;
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6);
    TFT.fillrect(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X - 1;
    y1 = CLOCK_CENTER_Y - (CLOCK_RADIUS - 6);
    x2 = CLOCK_CENTER_X + 1;
    y2 = CLOCK_CENTER_Y - (CLOCK_RADIUS - 16);
    TFT.fillrect(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 16);
    y1 = CLOCK_CENTER_Y - 1;
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6);
    y2 = CLOCK_CENTER_Y + 1;
    TFT.fillrect(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X - (CLOCK_RADIUS - 6);
    y1 = CLOCK_CENTER_Y - 1;
    x2 = CLOCK_CENTER_X - (CLOCK_RADIUS - 16);
    y2 = CLOCK_CENTER_Y + 1;
    TFT.fillrect(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 30)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 30)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 30)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 30)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 60)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 60)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 60)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 60)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 120)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 120)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 120)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 120)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 150)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 150)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 150)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 150)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 210)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 210)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 210)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 210)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 240)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 240)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 240)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 240)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 300)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 300)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 300)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 300)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

    x1 = CLOCK_CENTER_X + (CLOCK_RADIUS - 13) * sin(3.14 * ((double) 330)/180);
    y1 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 13) * cos(3.14 * ((double) 330)/180);
    x2 = CLOCK_CENTER_X + (CLOCK_RADIUS - 6) * sin(3.14 * ((double) 330)/180);
    y2 = CLOCK_CENTER_Y + (CLOCK_RADIUS - 6) * cos(3.14 * ((double) 330)/180);
    TFT.line(x1,y1,x2,y2,TEXTCOLOR);

}

// Map funktion copied from Arduino code
int map (int x, int in_min, int in_max, int out_min, int out_max)
{
    return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}