/* Water Meter HW driver for Water Meter Demo
 * 
 * Copyright (C) u-blox Melbourn Ltd
 * u-blox Melbourn Ltd, Melbourn, UK
 * 
 * All rights reserved.
 *
 * This source file is the sole property of u-blox Melbourn Ltd.
 * Reproduction or utilization of this source in whole or part is
 * forbidden without the written consent of u-blox Melbourn Ltd.
 */

#ifndef WATER_METER_API_HPP
#define WATER_METER_API_HPP

/**
 * @file water_meter_api.h
 * This file defines the API to the water meter HW
 * for the MWC demo 2015.
 */

// ----------------------------------------------------------------
// HARDWARE
// The code in this library is setup to expect a FlowIQ2100 water
// meter to be connected through this circuit:
//
// C027N board                                       Water Meter
//
//    5V   o-------------------o-------------o------o Red (pwr/clk)
//                             |             |
//                             |             |
//                             |             |
//                             |             |
//                             |             |
//                             |             |
//                             |             |
//                            | |            |
//                            | | 10k Ohm    |
//                            | |            |
//                             |             | 
//              1k Ohm         o-------------|------o Black (gnd)
//              ____          /              |
//     D4   o--|____|--------| (transistor)  |  
//                           _\/             |
//                             |            | |
//                             |            | | 10k Ohm
//                             |            | |
//                             |             |
//     D2   o--------------------------------o------o Green (data)
//                             |
//                             |
//    Gnd   o------------------o
//                      
// ----------------------------------------------------------------

// ----------------------------------------------------------------
// GENERAL COMPILE-TIME CONSTANTS
// ----------------------------------------------------------------

// ----------------------------------------------------------------
// CLASSES
// ----------------------------------------------------------------

class WaterMeterHandler {
public:
    /// Constructor.
    WaterMeterHandler (void);
    /// Destructor.
    ~WaterMeterHandler(void);
    /// Initialise the water meter, ensuring that there
    // is an electrical connection and that at least
    // a single character can be correctly received from
    // it (by checking parity).
    // \param data  pin to use for receiving data on the
    // C027N board.
    // \param pwrClk  pin to use to send pwr/clk on the
    // C027N board.
    // \param clkRateHz  the clock rate to use when
    // reading out data bits.
    // \return  true if successful, otherwise false.
    bool init (PinName dataPin = D2, PinName pwrClkPin = D4, uint32_t clkRateHz = 1000);

    /// Take a water volume reading from the meter.
    // \param  a pointer to somewhere to put the reading.
    // Only written-to if the outcome is true.  May be NULL,
    // in which case the success check is still carried out.
    // \\return  success if true, otherwise false.
    bool readLitres (uint32_t * pValue);

    /// Read the serial number from the meter.
    // \param  a pointer to somewhere to put the serial number.
    // Only written-to if the outcome is true.  May be NULL,
    // in which case the success check is still carried out.
    // \\return  success if true, otherwise false.
    bool readSerialNumber (uint32_t * pValue);

    /// Set debug to bool.
    // \param onNotOff debug will come out if this is true.
    // Default is no debug.
    void setDebugOn (bool onNotOff);

private:
    /// Set this to true once initialised.
    bool ready;
    /// Set this to true for debug.
    bool debug;
    /// The pin on the C027N board to use for data reception
    // from the meter.
    DigitalIn *pData;
    /// The pin on the C027N board to use for pwr/clock to
    // the meter.
    DigitalOut *pPwrClk;
    /// The length of half a clock period in microseconds to
    // use when reading data off the water meter.
    uint32_t halfClkPeriodUs;
    /// Power on the meter (by toggling the pwrClk line until
    // the data line goes low).
    // \return  true if successful, otherwise false.
    bool pwrOn (void);
    /// Power off the meter (by setting the pwrClk line low).
    void pwrOff (void);
    /// Read a start bit from the meter.
    // \return true if successful, otherwise false.
    bool readStartBit (void);
    /// Read a stop bit from the meter.
    // \return true if successful, otherwise false.
    bool readStopBit (void);
    /// Toggle the pwrClk line to read a data bit from the water
    // meter.
    // \return true if a 1 is read, false if a 0 is read.
    bool readDataBit (void);
    /// Check that parity is good
    // \return  true if successful, otherwise false.
    bool parityIsGood (uint8_t numOnes);
    /// Read a character from the meter and check parity.
    // \param pChar  a pointer to a location where the received
    // character can be stored.  Only filled in if the read is
    // successful.  May be NULL, in which case the parity check
    // is still carried out.
    // \return  true if successful, otherwise false.
    bool readChar (char *pChar);
    /// Read a whole fixed-length message from the water meter.
    // Reading stops when a parity error occurs or when the
    // requested number of bytes has been read.
    // \param pChars  a pointer to a location where the received
    // data bytes can be stored.  Must be at least numBytes big.
    // \return  the number of bytes read.
    // \param numChars  the number of bytes to read.
    uint32_t readFixedLengthMsg (char *pChars, uint32_t numChars);
    /// Convert a number of characters representing a decimal
    // value into a uint32_t value.
    // Characters are assumed to be ASCII, single byte values.
    // If a non-numeric character (including a decimal point)
    // is found then fail is returned.
    // \param pChars  a pointer to the start of the character array.
    // \param numChars  the number of characters.
    // \param pValue  the uint32_t value, only written-to if
    // success is returned.  May be NULL in which case the success
    // check is still carried out.
    // \return true if successful, otherwise false.
    bool decCharsToUint32 (char * pChars, uint32_t numChars, uint32_t * pValue);
};

#endif

// End Of File
