#include "mbed.h"

// Biblioteca do BME280: Temperatura e Pressão
#include "BME280.h"

// Biblioteca para utilização do RFID
#include "MFRC522.h"

// Declaração para utilização do serial bluetooth
Serial bluetooth(p9, p10);

// Declaração para utilização do sensor de temperatura e pressão
BME280 sensor_t_p(p28, p27);

// Declaração para utilização da bomba
PwmOut bomba(p21);

// Declaração para utilização do sensor de umidade
AnalogIn sensor_u(p15);

// Declaração para utilização do RFID
MFRC522 RFID(p11, p12, p13, p14, p8);

// Declaração para status de conexão do bluetooth + controle de login
DigitalIn statusBluetooth(p19);
int controleReconexao = 0;

// Variável de verificação de autenticação
int autenticado = 0;

// Variável de entrada bluetooth
char input;

// Controle manual da bomba
int controleManual = 0;


// Definição dos protótipos
void controleAutomatico();
void exibirMenu();
void exibirInfo();
void flushSerialBuffer();
void conexaoBluetooth();

// Exibição padrão do menu de controle
void exibirMenu()
{
    bluetooth.printf("\nSelecione a operacao desejada:\n");
    bluetooth.printf("(I)nformacoes de Pressao, Temperatura e Umidade\n");
    bluetooth.printf("(L)igar a Bomba\n");
    bluetooth.printf("(D)esligar a Bomba\n");
    bluetooth.printf("(S)air\n");
    controleAutomatico();
    conexaoBluetooth();
}

// Função de controle do status da conexão Bluetooth
void conexaoBluetooth()
{
    if (!statusBluetooth) {
        autenticado = 0;
        controleManual = 0;
        controleAutomatico();
        controleReconexao = 1;
    } 
    else {
        if (controleReconexao){
            wait(2.0f);
            bluetooth.printf("\n\nBem vindo ao sistema de controle da estufa.\nPara acessar as configuracoes, identifique-se.\n\n");
            controleReconexao = 0;
        }
    }
}

// Função para zerar a entrada depois da autenticação
void flushSerialBuffer()
{
    char char1 = 0;
    while (bluetooth.readable()) {
        char1 = bluetooth.getc();
    }
}


// Função de exibição dos sensores
void exibirInfo()
{
    int aux = 1;
    char sair;
    bluetooth.printf("Exibindo informacoes. 'S' para sair.\n");
    while (aux && autenticado) {
        bluetooth.printf("%2.2f C, %04.2f hPa, Umidade: %3.3f%%\n\n", sensor_t_p.getTemperature(), sensor_t_p.getPressure(), ((1-sensor_u.read()) * 100.0f)/0.6);
        controleAutomatico();
        conexaoBluetooth();

        if (bluetooth.readable()) {
            sair = bluetooth.getc();
            if (sair == 'S' || sair == 's') {
                bluetooth.printf("Rotina cancelada.\n");
                aux = 0;
                exibirMenu();
                continue;
            }
        }
        wait(5.0f);
    }
}

// Função do monitoramento da umidade da terra, para acionar/desativar a bomba
void controleAutomatico()
{
    if (!controleManual) {
        if (((1-sensor_u.read()) * 100.0f)/0.6 < 20.0) {
            bomba.write(0.2f);
        }

        if (((1-sensor_u.read()) * 100.0f)/0.6 > 40.0) {
            bomba.write(0.0f);
        }
    }
}

// Programa principal
int main()
{
    bomba.period_ms(10);
    bluetooth.baud(9600);
    RFID.PCD_Init();


    bluetooth.printf("\n\nBem vindo ao sistema de controle da estufa.\nPara acessar as configuracoes, identifique-se.\n\n");

    while(1) {
        controleAutomatico();
        conexaoBluetooth();

        // Programa a ser executado caso esteja autenticado
        if (autenticado) {
            exibirMenu();

            while (autenticado) {
                controleAutomatico();
                conexaoBluetooth();
                if (bluetooth.readable()) {

                    input = bluetooth.getc();

                    switch (input) {
                        case 'I':
                        case 'i':
                            exibirInfo();
                            break;

                        case 'L':
                        case 'l':
                            bomba.write(0.2f);
                            bluetooth.printf("Bomba acionada.\n");
                            controleManual = 1;
                            wait(1.0f);
                            exibirMenu();
                            break;

                        case 'D':
                        case 'd':
                            bomba.write(0.0f);
                            bluetooth.printf("Bomba desativada.\n");
                            controleManual = 0;
                            wait(1.0f);
                            exibirMenu();
                            break;

                        case 'S':
                        case 's':
                            autenticado = 0;
                            controleManual = 0;
                            controleAutomatico();
                            conexaoBluetooth();
                            bluetooth.printf("Desconectado com sucesso.\nPara logar novamente, aproxime o cracha RFID.\n\n");
                            break;
                    }
                }
            }
        }

        // Procura o card
        if ( ! RFID.PICC_IsNewCardPresent()) {
            wait_ms(500);
            continue;
        }

        // Seleciona um dos cards
        if ( ! RFID.PICC_ReadCardSerial()) {
            wait_ms(500);
            continue;
        }

        autenticado = 1;
        
        flushSerialBuffer();

        bluetooth.printf("Card reconhecido, acesso concedido.\n\n");

        wait(1.0f);

        // Imprime Card UID
        bluetooth.printf("Card UID: ");
        for (uint8_t i = 0; i < RFID.uid.size; i++) {
            bluetooth.printf(" %X02", RFID.uid.uidByte[i]);
        }
        bluetooth.printf("\n\n---------------------------------------------------------------------------\n");
        controleAutomatico();
        conexaoBluetooth();
        wait(1.0f);
    }
}
