/*
    A host controller driver from the mBed device.
    Copyright (C) 2012  Richard e Collins - richard.collins@linux.com

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <mbed.h>
#include <stddef.h>
#include <assert.h>

#include "UsbHostController.h"
#include "HardwareDefines.h"
#include "Debug.h"
#include "UsbStructures.h"
#include "UsbEnums.h"

#define HOST_CLK_EN        (1<<0)
#define PORTSEL_CLK_EN    (1<<3)
#define AHB_CLK_EN        (1<<4)
#define CLOCK_MASK        (HOST_CLK_EN | PORTSEL_CLK_EN | AHB_CLK_EN)

#define HOST_CONTROLLER_COMMUNICATIONS_AREA_ADDRESS (0x2007C000)

/*
 * The max number of connected devices I support.
 */
#define MAX_DEVICES 8

#if 0
#define DEBUGV DEBUG
#else
#define  DEBUGV(...)       do {} while(0)
#endif

namespace USB
{

/*
 * Does not matter where this is allocated, later will make it come after all the important stuff in the first block of extra sram so the lib takes no main ram from the user.
 */
static HostController _controller;

/**
 * Flash a light so I get too see stuff changing.
 */
static DigitalOut irqTick(LED3);
static DigitalOut errorLight(LED4);


/*
 * Handle interupts from the USB.
 * Some parts of this are taken from the official documentation but I was not !00% sure why
 * they did what they did. Lack of comments in their code or documentation.
 *
 * When the HC reaches the end of a frame and
 * its deferred interrupt register is 0, it writes the
 * current value of its HcDoneHead to this
 * location and generates an interrupt if interrupts
 * are enabled. This location is not written by the
 * HC again until software clears the WD bit in
 * the HcInterruptStatus register.
 * The LSb of this entry is set to 1 to indicate
 * whether an unmasked HcInterruptStatus was
 * set when HccaDoneHead was written.
 */
extern "C" void USB_IRQHandler(void) __irq;
void USB_IRQHandler (void) __irq
{    
    _controller.USBInterupt(); //Process the interupt.
}

/*
 * Called after we have recived a status changed intterupt from the USB.
 */
static void QueueRootHubStatusCheckCallback()
{
    DEBUGV("Resetting hub port after connect");
    //Reset the port ready for using the connected device.
    //ROOTHUB_PORT_RESET_STATUS_CHANGE bit will be set when done.
    LPC_USB->HcRhPortStatus1 = ROOTHUB_PORT_RESET_STATUS;
}

/**
 * Once started never stopped, as the memory is in reservers space there is nothing gained in stopping it.
 */
HostController *HostController::get()
{
    return &_controller;
}


/*
 * Process the interupt that just came in.
 */
uint32_t HostController::USBInterupt()
{
    // It is our interrupt, prevent HC from doing it to us again until we are finished
    LPC_USB->HcInterruptDisable = MasterInterruptEnable;
    
    //Other systems seem to grab the head and status fields, re enable interupts and then process whilst still in the handler, not sure why.
    TransferDescriptor *doneHead = _controller.commsArea->doneHead;
    _controller.commsArea->doneHead = 0;


    //Writing back the interupts we have handled tells the system we did them.
    //So for now say I done them all. ;)
    //If we don't do this the system locks up.
    //I guess because once we renable the interupts they refire so the main thread never gets a chance to runs again.
    //It is done via the return value to stop the compiler from optimising it out, which some may do if you are not carful.
    const uint32_t interruptStatus = LPC_USB->HcInterruptStatus;
    LPC_USB->HcInterruptStatus = interruptStatus;
    
    // It is our interrupt, prevent HC from doing it to us again until we are finished
    LPC_USB->HcInterruptEnable = MasterInterruptEnable;
    
    uint32_t servicedInterrupt = 0;
    if( interruptStatus & INTERRUPT_UNRECOVERABLE_ERROR )
    {
        flags.unrecoverableError = 1;
        errorLight = 1;
        servicedInterrupt |= INTERRUPT_UNRECOVERABLE_ERROR;
    }
    
    //Have we recived and data?
    if( interruptStatus & INTERRUPT_WRITEBACK_HEAD_DONE )
    {//Process all the recived packets.
        while( doneHead != NULL )
        {
//            DEBUG("TD 0x%08x done\n",(int)doneHead);
            
            //Get next before we start processing so the code can reuse the nextTD pointer.
            TransferDescriptor *next = doneHead->nextTD;
            
            //Tell driver transfer is done.
            messageCallback->TransferDone(doneHead);
            
            //If not zero then the call to messageCallback->TransferDone will queue the transfer for processing in the update phase.
            //The update call will then free the pool item.
            if( doneHead->transferCallback == NULL )
            {
                FreeMemoryPoolItem(doneHead);
                DEBUGV("TransferDescriptor freed");
            }
            //Move to next.
            doneHead = next;        
        }
    
        //All done re enable the interupt.
        servicedInterrupt |= INTERRUPT_WRITEBACK_HEAD_DONE;
    }
    
    if( interruptStatus & INTERRUPT_ROOTHUB_STATUS_CHANGE )
    {
        hubPortStatus = LPC_USB->HcRhPortStatus1;
        if( hubPortStatus & ROOTHUB_CONNECT_STATUS_CHANGE )
        {
            if(hubPortStatus & ROOTHUB_CURRENT_CONNECT_STATUS)
            {
                DEBUGV("Resetting hub port after connect");
                //Reset the port ready for using the connected device.
                //ROOTHUB_PORT_RESET_STATUS_CHANGE bit will be set when done.
                delayedRootHubStatusChangeTimeout.attach(QueueRootHubStatusCheckCallback,0.5f);
            }
            else
            {
                DEBUGV("Disconnected");
            }
        }
        else if( (hubPortStatus&(ROOTHUB_PORT_RESET_STATUS_CHANGE|ROOTHUB_PORT_RESET_STATUS)) == ROOTHUB_PORT_RESET_STATUS_CHANGE )
        {//Deal with port reset status.
            flags.newDeviceConnected = 1;
        }
        LPC_USB->HcRhPortStatus1 = hubPortStatus & 0xffff0000;

        servicedInterrupt |= INTERRUPT_ROOTHUB_STATUS_CHANGE;
    }
    
    if( interruptStatus & INTERRUPT_FRAME_NUMBER_OVERFLOW )
    {
        DEBUGV("FRAME_NUMBER_OVERFLOW");
        servicedInterrupt |= INTERRUPT_FRAME_NUMBER_OVERFLOW;
    }

    irqTick = !irqTick;//Show we got an interupt.
    
    return servicedInterrupt;
}

/*
 * Get it all going, I did want this to be done in the constructor but the chip did not like the USB starting before main was called.
 */
void HostController::Init(HostControllerMessages *messageCallback,Device*& deviceZero)
{
    this->messageCallback = messageCallback;

    errorLight = 0;
    commsArea = (HostControllerCommunicationsArea*)HOST_CONTROLLER_COMMUNICATIONS_AREA_ADDRESS;
    
    DEBUG("Starting USB Host Controller Driver.");
    DEBUG("sizeof(UsbHost::Controller) == %d",sizeof(HostController));
    DEBUG("sizeof(SetupPacket) == %d",sizeof(SetupPacket));
    DEBUG("%d %d %d %d %d",offsetof(SetupPacket,requestType),offsetof(SetupPacket,request),offsetof(SetupPacket,value),offsetof(SetupPacket,index),offsetof(SetupPacket,length));
    
    DEBUG("sizeof(HostControllerCommunicationsArea) == %d",sizeof(HostControllerCommunicationsArea));
    DEBUG("sizeof(EndpointDescriptor) == %d",sizeof(EndpointDescriptor));
    DEBUG("sizeof(TransferDescriptor) == %d",sizeof(TransferDescriptor));
    DEBUG("sizeof(LinkedListItem) == %d",sizeof(LinkedListItem));
    DEBUG("Memory pool size == %d",sizeof(commsArea->memoryPool));
    
    memset(commsArea,0,sizeof(HostControllerCommunicationsArea));
    
    deviceZero = &commsArea->deviceZero;
    
    //Init the memory pool, just creates a linked list of free objects.
    for( int n = 1 ; n < (sizeof(commsArea->memoryPool) / sizeof(commsArea->memoryPool[0])) ; n++ )
    {
        commsArea->memoryPool[n-1].next = commsArea->memoryPool + n;
    }
    memoryPoolHead = commsArea->memoryPool;
    numAllocatedPoolItems = 0;
        
    DEBUG("Disabling USB_IRQn");
    NVIC_DisableIRQ(USB_IRQn);
    
    // Turn on USB interface power/clock control bit in PCONP (Power Control for Peripherals register)
    LPC_SC->PCONP        |= PCUSB;//USB interface power/clock control bit. 
    
    // Enable USB host clock, port selection and AHB clock
    LPC_USB->USBClkCtrl |= CLOCK_MASK;
    
    // Wait for clocks to become available
    while ((LPC_USB->USBClkSt & CLOCK_MASK) != CLOCK_MASK);
    
    //    We are a Host
    LPC_USB->OTGStCtrl |= 1;
    LPC_USB->USBClkCtrl &= ~PORTSEL_CLK_EN;                // we don't need port selection clock until we do OTG

    // configure USB pins
    LPC_PINCON->PINSEL1 &= ~((3<<26)|(3<<28));    
    LPC_PINCON->PINSEL1 |=    ((1<<26)|(1<<28));            // USB D+/D-
    
    LPC_PINCON->PINSEL3 &= ~((3 << 6) | (3 << 22));        // USB_PPWR, USB_OVRCR
    LPC_PINCON->PINSEL3 |= ((2 << 6) | (2 << 22));
    
    LPC_PINCON->PINSEL4 &= ~(3 << 18);                    // USB_CONNECT
    LPC_PINCON->PINSEL4 |= (1 << 18);
    
    //    Reset OHCI block
    LPC_USB->HcControl         = 0;
    LPC_USB->HcControlHeadED = 0;
    LPC_USB->HcBulkHeadED     = 0;
    
    LPC_USB->HcCommandStatus = HostControllerReset;
    LPC_USB->HcFmInterval     = DEFAULT_FMINTERVAL;
    LPC_USB->HcPeriodicStart = FRAMEINTERVAL*90/100;
    
    LPC_USB->HcControl    = (LPC_USB->HcControl & (~HostControllerFunctionalState)) | OperationalMask;
    LPC_USB->HcRhStatus = SetGlobalPower;
    
    LPC_USB->HcHCCA = (uint32_t)commsArea;
    LPC_USB->HcInterruptStatus |= LPC_USB->HcInterruptStatus;
    LPC_USB->HcInterruptEnable = INTERRUPT_MASTER_INTERRUPT_ENABLE | INTERRUPT_WRITEBACK_HEAD_DONE | INTERRUPT_ROOTHUB_STATUS_CHANGE | INTERRUPT_UNRECOVERABLE_ERROR | INTERRUPT_FRAME_NUMBER_OVERFLOW;
    
    DEBUG("Enabling USB_IRQn");
    NVIC_SetPriority(USB_IRQn, 0);
    NVIC_EnableIRQ(USB_IRQn);
    
    wait(1);//Without this things can go a bit wonky because of the above playing with IRQ's. ( I think )
    
    DEBUG("Host controller communication area (HcHCCA) set to 0x%08x",LPC_USB->HcHCCA);
}

/*
 * Service the events from the interupts and the results of interupt transfers.
 */
void HostController::Update()
{
    irqTick = 0;//Clear this, may have been left on after an interupt.
    if( flags.newDeviceConnected )
    {
        flags.newDeviceConnected = 0;
        DEBUGV("AddDevice START");
        messageCallback->AddDevice(0,1,hubPortStatus);//Root hub is always hub zero, port 1. So add the device at this location.
        DEBUGV("AddDevice DONE");
    }
    
    if( flags.unrecoverableError )
    {
        DEBUG("USB Controller has failed, please reset.");
        flags.unrecoverableError = 0;
    }
}

void *HostController::AllocateMemoryPoolItem()
{
    if( memoryPoolHead == NULL )
    {
        return NULL;
    }
    LinkedListItem* f = memoryPoolHead;
    memoryPoolHead = memoryPoolHead->next;
    numAllocatedPoolItems++;
    DEBUGV("numAllocatedPoolItems (%d)",numAllocatedPoolItems);
    
    return (void*)f;
}
    
void HostController::FreeMemoryPoolItem(void *item)
{
    assert(numAllocatedPoolItems > 0);
    LinkedListItem* f = (LinkedListItem*)item;
    f->next = memoryPoolHead;
    memoryPoolHead = f;
    numAllocatedPoolItems--;
    DEBUGV("numAllocatedPoolItems (%d)",numAllocatedPoolItems);
}

/**
 * functionAddress  This is our deviceID. We tell the controller the ID we are using when it is inserted.
 *                  The docs call it the 'functionAddress'.
 * If the endpoint has need been used before it will not be linked into the list on the device. In that case we allocate it and add it.
 * When the device is removed all the enpoints are removed from the system.
 */
EndpointDescriptor *HostController::GetEndpointDescriptor(EndpointDescriptor** headEP,uint32_t functionAddress,int endpointNumber,int maximumPacketSize,int lowspeed)
{
    uint32_t control = (maximumPacketSize<<16) | (lowspeed<<13) | (endpointNumber<<7) | functionAddress;
    
    //have we already allocated and linked in the end point?
    EndpointDescriptor* ep = *headEP;    
    while( ep != NULL )
    {
        if( ep->control == control )
        {
            return ep;
        }
        ep = ep->NextEP;
    }

    //Get here and we have not see the endpoint before.
    //So we need to allocate it and link it in.
    ep = (EndpointDescriptor*)AllocateMemoryPoolItem();
    
    if( ep == NULL )
    {
        DEBUGV("AllocateEndpoint failed");
        return NULL;
    }
    
    //Link the endpoint onto the list of things to do the next frame when ControlListFilled of HcCommandStatus is set.    
    //When the host controller reaches the end of the list pointed to by HcControlCurrentED
    //It checks the ControlListEnable flag, if set then HcControlHeadED is copied to HcControlCurrentED and starts the new transfers.
    ep->NextEP = *headEP;
    *headEP = ep;

    ep->control = control;//Set the address etc...
    ep->TailTD = 0;
    ep->HeadTD = 0;
    ep->C = 0;
    ep->H = 0;
    ep->ZERO = 0;
    
    DEBUGV("endpoint allocated ep(%d) deviceID(%d)",endpointNumber,functionAddress);

    return ep;
}


TransferDescriptor *HostController::AllocateTransferDescriptor(int deviceID,int direction,const uint8_t* data,int dataLength)
{
    TransferDescriptor* td = (TransferDescriptor*)AllocateMemoryPoolItem();
    
    td->deviceID = (uint8_t)deviceID;//So we know who sent this transfer.
    td->bufferRounding = 1;//No rounding needed
    td->direction = direction;
    td->dataToggle = 0;
    td->conditionCode = 15;
    if( data != NULL )
    {
        td->CurrentBufferPointer = data;
        td->bufferEnd = (data + dataLength - 1);//Points to the last byte.
    }
    DEBUGV("transfer desc allocated %08x",*((uint32_t*)td));
    return td;
}

void HostController::QueueControlTransfer(int deviceID,int direction,int endpointNumber,int maximumPacketSize,int lowspeed,int dataToggle,const uint8_t* data,int dataLength)
{
    TransferDescriptor* td = QueueTransfer((EndpointDescriptor**)&LPC_USB->HcControlHeadED,deviceID,direction,endpointNumber,maximumPacketSize,lowspeed,data,dataLength);
    
   //If endpoint zero then toggle is always DATA0 for setup packet and DATA1 for the rest.
    if( endpointNumber == 0 )
    {
        dataToggle = direction == TDD_SETUP ? TOGGLE_DATA0 : TOGGLE_DATA1;
    }
    td->dataToggle = dataToggle;//DATA0 and use this field for the toggle value. MSb states use this filed, LSb is zero as it's DATA0
    td->transferType = TT_CONTROL;//For the driver.
    td->transferCallback = NULL;
    LPC_USB->HcCommandStatus = LPC_USB->HcCommandStatus | ControlListFilled;//Tell the host of new things to do.
    LPC_USB->HcControl = LPC_USB->HcControl | ControlListEnable;//Make sure list processing is on.
} 

void HostController::QueueBulkTransfer(int deviceID,int direction,int endpointNumber,int maximumPacketSize,int lowspeed,TransferCallback* callback,const uint8_t* data,int dataLength)
{
    TransferDescriptor* td = QueueTransfer((EndpointDescriptor**)&LPC_USB->HcBulkHeadED,deviceID,direction,endpointNumber,maximumPacketSize,lowspeed,data,dataLength);
    DEBUGV("QueueBulkTransfer (0x%08x)",td);
    
    //For transfers that need a callback.
    //If callbackID is non zero then when the transfer is done a callback is generated.
    td->transferCallback = callback;
    td->data = data;
    td->length = dataLength;
    td->transferType = TT_BULK;//For the driver.

    LPC_USB->HcCommandStatus = LPC_USB->HcCommandStatus | BulkListFilled;//Tell the host of new things to do.
    LPC_USB->HcControl = LPC_USB->HcControl | BulkListEnable;//Make sure list processing is on.
}

TransferDescriptor* HostController::QueueTransfer(EndpointDescriptor** headED,int deviceID,int direction,int endpointNumber,int maximumPacketSize,int lowspeed,const uint8_t* data,int dataLength)
{
    //Additions into transfer queues are always done by copying the new entry information to the entry
    //at the tail of the queue and then appending a new tail entry to the queue. This is accomplished
    //by:
    //  1. Copying the new information to the entry pointed to by TailP
    //  2. Setting the NextTD pointer in the current tail entry to a new place holder
    //  3. Advancing the TailP pointer to the new place holder
    //  4. Writing to the ControlListFilled or BulkListFilled bit in HcCommandStatus if the
    //      insert was to a queue on the Control list or Bulk list.

    TransferDescriptor* td = AllocateTransferDescriptor(deviceID,direction,data,dataLength);
    td->ep = endpointNumber;
    
    //Get the endpoint needed for this transfer.
    EndpointDescriptor* ep = GetEndpointDescriptor(headED,deviceID,endpointNumber,maximumPacketSize,lowspeed);
    
    //Can only queue one at a time, I need to fix.    
    ep->HeadTD = ((int)td)>>4;
    ep->TailTD = 0;
    td->nextTD = 0;
    
    return td;
}


uint8_t* HostController::getScratchRam()
{
    return commsArea->scratchRam;
}

void HostController::DetachDevice(int deviceID)
{
    RemoveEndpointsForDevice(deviceID,(EndpointDescriptor**)&LPC_USB->HcControlHeadED);
    RemoveEndpointsForDevice(deviceID,(EndpointDescriptor**)&LPC_USB->HcBulkHeadED);
    for( int n = 0 ; n < 32 ; n++ )
    {
        RemoveEndpointsForDevice(deviceID,(EndpointDescriptor**)&commsArea->interruptTable[n]);
    }
}

void HostController::RemoveEndpointsForDevice(int deviceID,EndpointDescriptor **list)
{
    while( (*list) != NULL )
    {
        if( ( (*(list))->control&0x7f) == deviceID )
        {
            EndpointDescriptor* old = *list;
            *list = old->NextEP;//Unlink old.
            //Should not be any transfers to do this EP if we are removing it...... TODO Check this!
            if( old->HeadTD != 0 )
            {
                DEBUGV("Removing endpoint with unfinished transfers, please fix bug richard!");
            }
            DEBUGV("Freed EP for device(%d)",deviceID);
            FreeMemoryPoolItem(old);
        }
        else
        {
            list = &(*(list))->NextEP;//Move to next.
        }
    }
}

};//namespace USB

