/*
    A host controller driver from the mBed device.
    Copyright (C) 2012  Richard e Collins - richard.collins@linux.com

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __HARDWARE_DEFINES_H__
#define __HARDWARE_DEFINES_H__

//USB interface power/clock control bit in PCONP (Power Control for Peripherals register)
#define PCUSB (1U<<31)


/**
 * HcInterruptStatus Register
 * This register provides status on various events that cause hardware interrupts. When an event
 * occurs, Host Controller sets the corresponding bit in this register. When a bit becomes set, a
 * hardware interrupt is generated if the interrupt is enabled in the HcInterruptEnable register (see
 * Section 7.1.5) and the MasterInterruptEnable bit is set. The Host Controller Driver may clear
 * specific bits in this register by writing &#65533;1&#65533; to bit positions to be cleared. The Host Controller
 * Driver may not set any of these bits. The Host Controller will never clear the bit.
 *
 * This bit is set when the USB schedule for the current Frame
 * overruns and after the update of HccaFrameNumber. A
 * scheduling overrun will also cause the
 * SchedulingOverrunCount of HcCommandStatus to be
 * incremented.
 */
#define INTERRUPT_SCHEDULING_OVERRUN (1U<<0)

/**
 * This bit is set immediately after HC has written HcDoneHead to
 * HccaDoneHead. Further updates of the HccaDoneHead will not
 * occur until this bit has been cleared. HCD should only clear this
 * bit after it has saved the content of HccaDoneHead.
 */
#define INTERRUPT_WRITEBACK_HEAD_DONE (1U<<1)

/**
 * This bit is set by HC at each start of a frame and after the
 * update of HccaFrameNumber. HC also generates a SOF token
 * at the same time.
 */
 #define INTERRUPT_START_OF_FRAME (1U<<2)

/**
 * This bit is set when HC detects that a device on the USB is
 * asserting resume signaling. It is the transition from no resume
 * signaling to resume signaling causing this bit to be set. This bit
 * is not set when HCD sets the USBRESUME state.
 */
 #define INTERRUPT_RESUME_DETECTED (1U<<3)

/**
 * This bit is set when HC detects a system error not related to
 * USB. HC should not proceed with any processing nor signaling
 * before the system error has been corrected. HCD clears this bit
 * after HC has been reset.
 */
 #define INTERRUPT_UNRECOVERABLE_ERROR (1U<<4)

/**
 * This bit is set when the MSb of HcFmNumber (bit 15) changes
 * value, from 0 to 1 or from 1 to 0, and after HccaFrameNumber
 * has been updated.
 */
 #define INTERRUPT_FRAME_NUMBER_OVERFLOW (1U<<5)

/**
 * This bit is set when the content of HcRhStatus or the content of
 * any of HcRhPortStatus[NumberofDownstreamPort] has
 * changed.
 */
 #define INTERRUPT_ROOTHUB_STATUS_CHANGE (1U<<6)

/**
 * This bit is set by HC when HCD sets the
 * OwnershipChangeRequest field in HcCommandStatus. This
 * event, when unmasked, will always generate an System
 * Management Interrupt (SMI) immediately.
 * This bit is tied to 0b when the SMI pin is not implemented.
 */
#define INTERRUPT_OWNERSHIP_CHANGED (1U<<30)

/**
 * Each enable bit in the HcInterruptEnable register corresponds to an associated interrupt bit in the
 * HcInterruptStatus register. The HcInterruptEnable register is used to control which events
 * generate a hardware interrupt. When a bit is set in the HcInterruptStatus register AND the
 * corresponding bit in the HcInterruptEnable register is set AND the MasterInterruptEnable bit is
 * set, then a hardware interrupt is requested on the host bus.
 * Writing a '1' to a bit in this register sets the corresponding bit, whereas writing a '0' to a bit in this
 * register leaves the corresponding bit unchanged. On read, the current value of this register is
 * returned.
 *
 * 
 * A &#65533;0&#65533; written to this field is ignored by HC. A '1' written to this
 * field enables interrupt generation due to events specified in the
 * other bits of this register. This is used by HCD as a Master Interrupt Enable.
 */
#define INTERRUPT_MASTER_INTERRUPT_ENABLE (1U<<31) 


/**
 * HcRhPortStatus[1:NDP] Register  ROOT HUB, mBed chip only has one.
 * The HcRhPortStatus[1:NDP] register is used to control and report port events on a per-port
 * basis. NumberDownstreamPorts represents the number of HcRhPortStatus registers that are
 * implemented in hardware. The lower word is used to reflect the port status, whereas the upper
 * word reflects the status change bits. Some status bits are implemented with special write behavior
 * (see below). If a transaction (token through handshake) is in progress when a write to change
 * port status occurs, the resulting port status change must be postponed until the transaction
 * completes. Reserved bits should always be written '0'.
 */
 
 /**
  * This bit reflects the current state of the downstream port.
  * 0 = no device connected
  * 1 = device connected
  * (write) ClearPortEnable
  * The HCD writes a &#65533;1&#65533; to this bit to clear the PortEnableStatus bit.
  * Writing a &#65533;0&#65533; has no effect. The CurrentConnectStatus is not
  * affected by any write.
  * Note: This bit is always read &#65533;1b&#65533; when the attached device is
  * nonremovable (DeviceRemoveable[NDP]).
  */
 #define ROOTHUB_CURRENT_CONNECT_STATUS (1U<<0)
 
 /**
  * (read) PortResetStatus
  * When this bit is set by a write to SetPortReset, port reset
  * signaling is asserted. When reset is completed, this bit is cleared
  * when PortResetStatusChange is set. This bit cannot be set if
  * CurrentConnectStatus is cleared.
  * 0 = port reset signal is not active
  * 1 = port reset signal is active
  *
  * (write) SetPortReset
  * The HCD sets the port reset signaling by writing a &#65533;1&#65533; to this bit.
  * Writing a &#65533;0&#65533; has no effect. If CurrentConnectStatus is cleared,
  * this write does not set PortResetStatus, but instead sets
  * ConnectStatusChange. This informs the driver that it attempted
  * to reset a disconnected port.
  */
#define ROOTHUB_PORT_RESET_STATUS (1U<<4)
 
/**
 * (read) LowSpeedDeviceAttached
 * This bit indicates the speed of the device attached to this port.
 * When set, a Low Speed device is attached to this port. When
 * clear, a Full Speed device is attached to this port. This field is
 * valid only when the CurrentConnectStatus is set.
 * 0 = full speed device attached
 * 1 = low speed device attached
 *
 * (write) ClearPortPower
 * The HCD clears the PortPowerStatus bit by writing a &#65533;1&#65533; to this
 * bit. Writing a &#65533;0&#65533; has no effect.
 */ 
#define ROOTHUB_LOW_SPEED_DEVICE_ATTACHED (1U<<9)

 /**
  * ConnectStatusChange.
  * This bit is set whenever a connect or disconnect event occurs.
  * The HCD writes a &#65533;1&#65533; to clear this bit. Writing a &#65533;0&#65533; has no effect. If
  * CurrentConnectStatus is cleared when a SetPortReset,
  * SetPortEnable, or SetPortSuspend write occurs, this bit is set to
  * force the driver to re-evaluate the connection status since these
  * writes should not occur if the port is disconnected.
  * 0 = no change in CurrentConnectStatus
  * 1 = change in CurrentConnectStatus
  * Note: If the DeviceRemovable[NDP] bit is set, this bit is set only
  * after a Root Hub reset to inform the system that the device is
  * attached.
 */
 #define ROOTHUB_CONNECT_STATUS_CHANGE (1U<<16)
 
 /**
  * PortResetStatusChange
  * This bit is set at the end of the 10-ms port reset signal.
  * The HCD writes a &#65533;1&#65533; to clear this bit. Writing a &#65533;0&#65533; has no effect.
  * 0 = port reset is not complete
  * 1 = port reset is complete
  */
 #define ROOTHUB_PORT_RESET_STATUS_CHANGE (1U<<20)
 
 
//===================================================================
// Taken from USBHost.cpp written by Copyright (c) 2010 Peter Barrett
//===================================================================
//    Hardware defines

//    HcControl

#define    ControlListEnable    0x00000010
#define    BulkListEnable        0x00000020

#define    OperationalMask        0x00000080
#define    HostControllerFunctionalState    0x000000C0

//    HcCommandStatus
#define HostControllerReset    0x00000001
#define ControlListFilled    0x00000002
#define BulkListFilled        0x00000004

//    HcInterruptStatus Register
#define MasterInterruptEnable    0x80000000

//    HcRhStatus
#define SetGlobalPower            0x00010000

//For the clock stuff.
#define  FRAMEINTERVAL        (12000-1)    // 1ms
#define  DEFAULT_FMINTERVAL    ((((6 * (FRAMEINTERVAL - 210)) / 7) << 16) | FRAMEINTERVAL)


#endif //__HARDWARE_DEFINES_H__
