#include "mbed.h"
#include "mbed_rpc.h"
#include "SerialRPCInterface.h"

enum command_t{ //MOTORCONTROLLER COMMANDS
                    SET_MODE = 1,
                    SET_CONSTANT_PRESSURE, 
                    SET_CONSTANT_FLOW, 
                    SET_CONSTANT_SPEED, 
                    SET_MIN,
                    SET_MAX, 
                    SET_FREQUENCY, 
                    RECEIVE_PRESSURE, 
                    RECEIVE_FLOW,
                //SENSORCONTROLLER COMMANDS
                    //Pressure sensor commands
                    SET_SENSOR_PRESSURE_1_SAMPLE_RATE = 13,
                    SET_SENSOR_PRESSURE_1_MVA,
                    SET_RESPONSE_SENSOR_PRESSURE_1,
                    SET_SENSOR_PRESSURE_2_SAMPLE_RATE,
                    SET_SENSOR_PRESSURE_2_MVA,
                    SET_RESPONSE_SENSOR_PRESSURE_2,
                    //Temperature sensor commands
                    SET_SENSOR_TEMPERATURE_1_SAMPLE_RATE,
                    SET_SENSOR_TEMPERATURE_1_MVA,
                    SET_RESPONSE_SENSOR_TEMPERATURE_1,
                    SET_SENSOR_TEMPERATURE_2_SAMPLE_RATE,
                    SET_SENSOR_TEMPERATURE_2_MVA,
                    SET_RESPONSE_SENSOR_TEMPERATURE_2,
                    //Flow sensor commands
                    SET_SENSOR_FLOW_SAMPLE_RATE,
                    SET_SENSOR_FLOW_MVA,
                    SET_RESPONSE_SENSOR_FLOW,
                    //Motor sensor
                    //Note: currently not used
                    SET_SENSOR_SPEED_SAMPLE_RATE,
                    SET_SENSOR_SPEED_MVA,
                    SET_RESPONSE_SENSOR_SPEED
                } command;


//SENSOR OUTPUTS
    float sensor_pressure_1 = 0;
    float sensor_pressure_2 = 0;
    float sensor_temperature_1 = 0;
    float sensor_temperature_2 = 0;
    float sensor_flow = 0;
    
    RPCVariable<float>  RPCsensor_pressure_1(&sensor_pressure_1, "sensor_pressure_1");
    RPCVariable<float>  RPCsensor_pressure_2(&sensor_pressure_2, "sensor_pressure_2");
    RPCVariable<float>  RPCsensor_temperature_1(&sensor_temperature_1, "sensor_temperature_1");
    RPCVariable<float>  RPCsensor_temperature_2(&sensor_temperature_2, "sensor_temperature_2");
    RPCVariable<float>  RPCsensor_flow(&sensor_flow, "sensor_flow");

//CONTROLLER SETTINGS
    //SENSORCONTROLLER - MOVING AVERAGE
    int druksensor1_mva_elements = 10,
        druksensor2_mva_elements = 10,
        tempsensor1_mva_elements = 10,
        tempsensor2_mva_elements = 10,
        flowsensor_mva_elements = 10;

    RPCVariable<int>  RPCdruksensor1_mva_elements(&druksensor1_mva_elements, "druksensor1_mva_elements");
    RPCVariable<int>  RPCdruksensor2_mva_elements(&druksensor2_mva_elements, "druksensor2_mva_elements");
    RPCVariable<int>  RPCtempsensor1_mva_elements(&tempsensor1_mva_elements, "tempsensor1_mva_elements");
    RPCVariable<int>  RPCtempsensor2_mva_elements(&tempsensor2_mva_elements, "tempsensor2_mva_elements");
    RPCVariable<int>  RPCflowsensor_mva_elements(&flowsensor_mva_elements, "flowsensor_mva_elements");
    //MOTORCONTROLLER - SETTINGS
    int motorcontroller_mode = 0,
        motorcontroller_constant_pressure = 0,
        motorcontroller_constant_flow = 0,
        motorcontroller_constant_speed = 0,
        motorcontroller_min = 0,
        motorcontroller_max = 0,
        motorcontroller_frequency = 0;
    
    RPCVariable<int>  RPCmotorcontroller_mode(&motorcontroller_mode, "motorcontroller_mode");
    RPCVariable<int>  RPCmotorcontroller_constant_pressure(&motorcontroller_constant_pressure, "motorcontroller_constant_pressure");
    RPCVariable<int>  RPCmotorcontroller_constant_flow(&motorcontroller_constant_flow, "motorcontroller_constant_flow");
    RPCVariable<int>  RPCmotorcontroller_constant_speed(&motorcontroller_constant_speed, "motorcontroller_constant_speed");
    RPCVariable<int>  RPCmotorcontroller_min(&motorcontroller_min, "motorcontroller_min");
    RPCVariable<int>  RPCmotorcontroller_max(&motorcontroller_max, "motorcontroller_max");
    RPCVariable<int>  RPCmotorcontroller_frequency(&motorcontroller_frequency, "motorcontroller_frequency");   
    
//UPDATE PERIODS
    float pressure1_update_period_s = 1;
    float pressure2_update_period_s = 1;
    float temperature1_update_period_s = 1;
    float temperature2_update_period_s = 1;
    float flow_update_period_s = 1;
    
    RPCVariable<float>  RPCpressure1_update_period_s(&pressure1_update_period_s, "pressure1_update_period_s");
    RPCVariable<float>  RPCpressure2_update_period_s(&pressure2_update_period_s, "pressure2_update_period_s");
    RPCVariable<float>  RPCtemperature1_update_period_s(&temperature1_update_period_s, "temperature1_update_period_s");
    RPCVariable<float>  RPCtemperature2_update_period_s(&temperature2_update_period_s, "temperature2_update_period_s");
    RPCVariable<float>  RPCflow_update_period_s(&flow_update_period_s, "flow_update_period_s");

Serial pc(USBTX, USBRX); //Enable USB communication


//I2C settings
#define SDA D14
#define SCL D15
#define I2C_BUFFER_SIZE 10
I2C master(SDA,SCL);

//Controller addresses
#define motor_addr 0x91
#define interface_addr 0x92
#define sensor_addr 0x93



//Handles all the RPC communication. 
//Taken from the example at developer.mbed.org/cookbook
void RPC_routine(){
    static char buf[256], outbuf[256];
    pc.gets(buf, 256);
    //Call the static call method on the RPC class
    RPC::call(buf, outbuf); 
    pc.printf("%s\n", outbuf);
}
//Split an integer into two char
void int_to_2char(char* arr, int val, int first_element = 0){
    arr[first_element] = val>>8;
    arr[first_element+1] = val&255;
}
//Join two char to make an integer. 
int char2_to_int(char* arr, int first_element = 0){
    return (arr[first_element]<<8)+arr[first_element+1];
}

void updatePressure1(){
    //Request data from sensor controller
    //Immediately send it back to the motor controller
    char buffer[3] = {0};
    buffer[0] = SET_RESPONSE_SENSOR_PRESSURE_1;
    master.write(sensor_addr,buffer,1);
    buffer[0] = 0; //clear buffer again
    master.read(sensor_addr,buffer,2);
    //Update the pressure value for LabView
    sensor_pressure_1 = char2_to_int(buffer,0); 
    //Move the received data over one spot to make place for a command byte
    buffer[2] = buffer[1];
    buffer[1] = buffer[0];
    buffer[0] = RECEIVE_PRESSURE;
    //Update motor controller
    master.write(motor_addr,buffer,2);
}

void updatePressure2(){
    //For comments see 'updatePressure1()'
    char buffer[3] = {0};
    buffer[0] = SET_RESPONSE_SENSOR_PRESSURE_2;
    master.write(sensor_addr,buffer,1);
    buffer[0] = 0;
    master.read(sensor_addr,buffer,2);
    sensor_pressure_2 = char2_to_int(buffer,0);    
}


void updateTemperature1(){
    //For comments see 'updatePressure1()'
    char buffer[3] = {0};
    buffer[0] = SET_RESPONSE_SENSOR_TEMPERATURE_1;
    master.write(sensor_addr,buffer,1);
    buffer[0] = 0;
    master.read(sensor_addr,buffer,2);
    sensor_temperature_1 = char2_to_int(buffer,0);   
}

void updateTemperature2(){
    //For comments see 'updatePressure1()'
    char buffer[3] = {0};
    buffer[0] = SET_RESPONSE_SENSOR_TEMPERATURE_2;
    master.write(sensor_addr,buffer,1);
    buffer[0] = 0;
    master.read(sensor_addr,buffer,2);
    sensor_temperature_2 = char2_to_int(buffer,0);   
}

void updateFlow(){
    //For comments see 'updatePressure1()'
    char buffer[3] = {0};
    buffer[0] = SET_RESPONSE_SENSOR_FLOW;
    master.write(sensor_addr,buffer,1);
    buffer[0] = 0;
    master.read(sensor_addr,buffer,2);
    sensor_flow = char2_to_int(buffer,0); 
    buffer[2] = buffer[1];
    buffer[1] = buffer[0];
    buffer[0] = RECEIVE_FLOW;
    master.write(motor_addr,buffer,2);    
}

Timer t;

Ticker  tick_pressure1,
        tick_pressure2,
        tick_temperature1,
        tick_temperature2,
        tick_flow;

void update_helper(int addr, command_t cmd, int val){
    char buffer[3] = {0};
    buffer[0] = cmd;
    int_to_2char(buffer, val, 1);
    master.write(addr, buffer, 3); 
}

//DANGER:   WEAR EYE PROTECTION AT ALL TIMES
//          THIS FUNCTION IS CLASS 1 UGLY
//          POTENTIAL OF EYE BURNING.
void updateI2Cstuff(){
    //Only updates when the variables has actually changed
    
    //Sensor controller read rate
    static float old_pressure1_update_period_s          = 0,
                old_pressure2_update_period_s           = 0,
                old_temperature1_update_period_s        = 0,
                old_temperature2_update_period_s        = 0,
                old_flow_update_period_s                = 0;
    //Sensor controller smoothing factor                
    static int  old_druksensor1_mva_elements            = 0,
                old_druksensor2_mva_elements            = 0,
                old_tempsensor1_mva_elements            = 0,
                old_tempsensor2_mva_elements            = 0,
                old_flowsensor_mva_elements             = 0;
    //Motor controller settings              
    static int  old_motorcontroller_mode                = 0,
                old_motorcontroller_constant_pressure   = 0,
                old_motorcontroller_constant_flow       = 0,
                old_motorcontroller_constant_speed      = 0,
                old_motorcontroller_min                 = 0,
                old_motorcontroller_max                 = 0,
                old_motorcontroller_frequency           = 0;
                
    if(old_pressure1_update_period_s != pressure1_update_period_s){
        tick_pressure1.attach(&updatePressure1,pressure1_update_period_s);
        old_pressure1_update_period_s = pressure1_update_period_s;
    }
    if(old_pressure2_update_period_s != pressure2_update_period_s){
        tick_pressure2.attach(&updatePressure2,pressure2_update_period_s);
        old_pressure2_update_period_s = pressure2_update_period_s;
    }
    if(old_temperature1_update_period_s != temperature1_update_period_s){
        tick_temperature1.attach(&updateTemperature1,temperature1_update_period_s);
        old_temperature1_update_period_s = temperature1_update_period_s;
    }
    if(old_temperature2_update_period_s != temperature2_update_period_s){
        tick_temperature2.attach(&updateTemperature2,temperature2_update_period_s);
        old_temperature2_update_period_s = temperature2_update_period_s;
    }
    if(old_flow_update_period_s != flow_update_period_s){
        tick_flow.attach(&updateFlow,flow_update_period_s);  
        old_flow_update_period_s = flow_update_period_s;
    }
    if(old_druksensor1_mva_elements != druksensor1_mva_elements){
        update_helper(sensor_addr, SET_SENSOR_PRESSURE_1_MVA, druksensor1_mva_elements);
        old_druksensor1_mva_elements = druksensor1_mva_elements;
    }
    if(old_druksensor2_mva_elements != druksensor2_mva_elements){
        update_helper(sensor_addr, SET_SENSOR_PRESSURE_2_MVA, druksensor2_mva_elements);
        old_druksensor2_mva_elements = druksensor2_mva_elements;
    }
    if(old_tempsensor1_mva_elements != tempsensor1_mva_elements){
        update_helper(sensor_addr, SET_SENSOR_TEMPERATURE_1_MVA, tempsensor1_mva_elements);        
        old_tempsensor1_mva_elements = tempsensor1_mva_elements;
    }
    if(old_tempsensor2_mva_elements != tempsensor2_mva_elements){
        update_helper(sensor_addr, SET_SENSOR_TEMPERATURE_2_MVA, tempsensor2_mva_elements);
        old_tempsensor2_mva_elements = tempsensor2_mva_elements;
    }
    if(old_flowsensor_mva_elements != flowsensor_mva_elements){
        update_helper(sensor_addr, SET_SENSOR_FLOW_MVA, flowsensor_mva_elements);
        old_flowsensor_mva_elements = flowsensor_mva_elements;
    }
    
    //Motorcontroller
    if(old_motorcontroller_mode != motorcontroller_mode){
        update_helper(motor_addr, SET_MODE, motorcontroller_mode);
        old_motorcontroller_mode = motorcontroller_mode;
    }
    if(old_motorcontroller_constant_pressure != motorcontroller_constant_pressure){
        update_helper(motor_addr, SET_CONSTANT_PRESSURE, motorcontroller_constant_pressure);
        old_motorcontroller_constant_pressure = motorcontroller_mode;
    }
    if(old_motorcontroller_constant_flow != motorcontroller_constant_flow){
        update_helper(motor_addr, SET_CONSTANT_FLOW, motorcontroller_constant_flow);
        old_motorcontroller_constant_flow = motorcontroller_constant_flow;
    }
    if(old_motorcontroller_constant_speed != motorcontroller_constant_speed){
        update_helper(motor_addr, SET_CONSTANT_SPEED, motorcontroller_constant_speed);
        old_motorcontroller_constant_speed = old_motorcontroller_constant_speed;
    }
    if(old_motorcontroller_min != motorcontroller_min){
        update_helper(motor_addr, SET_MIN, motorcontroller_min);
        old_motorcontroller_min = motorcontroller_min;
    }
    if(old_motorcontroller_max != motorcontroller_max){
        update_helper(motor_addr, SET_MAX, motorcontroller_max);
        old_motorcontroller_max = motorcontroller_max;
    }
    if(old_motorcontroller_frequency != motorcontroller_frequency){
        update_helper(motor_addr, SET_FREQUENCY, motorcontroller_frequency);
        old_motorcontroller_frequency = motorcontroller_frequency;
    }                         
}

int main() {
    t.start();  
    while(1) {
        RPC_routine();
        if(t.read() > 1){
            //Only update the tickers every second  
            updateI2Cstuff();
        }
        wait_ms(1);
    }
}