/*
 * SOURCE FILE : main.cpp
 *
 * Test program to test out WiiClassicController class.
 *
 */
 
#include "mbed.h"
#include "WiiClassicControllerWithCalibration.h"

// Define this to dump packet in hex.
#undef DUMP_PACKET

#define LOOP_DELAY (float)0.025 // seconds

// global declarations
Serial pc(USBTX, USBRX);

/* Read from the Wii classic controller and display results on serial port output.
 * @param ctrlr Controller to read from.
 * @param portname Name of port you are reading from.
 */
static void ReadAndReport( WiiClassicControllerWithCalibration *ctrlr, const char *portname ) {
    static int counter = 0;
    if( ctrlr->Read() ) {

        // Display updated every tenth time.    
        if( counter == 0 ) {

            pc.printf( "%c[H", 27 );                            // VT100 home cursor
            pc.printf("%s\r\n", portname);
        
            #ifdef DUMP_PACKET
                int bufSize = ctrlr->GetReadBufSize();
                UInt8 *bufPtr = ctrlr->GetReadBuf();
                for (int i = 0; i < bufSize; i++) {
                    pc.printf("%x ", bufPtr[i]);
                }
                pc.printf("\r\n");
            #endif
            
            pc.printf("A=%u\r\n", ctrlr->GetButtonA());
            pc.printf("B=%u\r\n", ctrlr->GetButtonB());
            pc.printf("X=%u\r\n", ctrlr->GetButtonX());
            pc.printf("Y=%u\r\n", ctrlr->GetButtonY());
            pc.printf("ZL=%u\r\n", ctrlr->GetButtonZL());
            pc.printf("ZR=%u\r\n", ctrlr->GetButtonZR());
            pc.printf("Up=%u\r\n", ctrlr->GetButtonUp());
            pc.printf("Down=%u\r\n", ctrlr->GetButtonDown());
            pc.printf("Left=%u\r\n", ctrlr->GetButtonLeft());
            pc.printf("Right=%u\r\n", ctrlr->GetButtonRight());
            pc.printf("Home=%u\r\n", ctrlr->GetButtonHome());
            pc.printf("Select=%u\r\n", ctrlr->GetButtonSelect());
            pc.printf("Start=%u\r\n", ctrlr->GetButtonStart());
            pc.printf("LeftTrigger=%2u (%6.3f)\r\n", ctrlr->GetLeftTrigger(), ctrlr->GetCalLeftTrigger() );
            pc.printf("ButtonLT=%u\r\n", ctrlr->GetButtonLT());
            pc.printf("RightTrigger=%2u (%6.3f)\r\n", ctrlr->GetRightTrigger(), ctrlr->GetCalRightTrigger() );
            pc.printf("ButtonRT=%u\r\n", ctrlr->GetButtonRT());
            pc.printf("LJoyX=%2u (%6.3f)\r\n", ctrlr->GetLJoyX(), ctrlr->GetCalLJoyX() );
            pc.printf("LJoyY=%2u (%6.3f)\r\n", ctrlr->GetLJoyY(), ctrlr->GetCalLJoyY() );
            pc.printf("RJoyX=%2u (%6.3f)\r\n", ctrlr->GetRJoyX(), ctrlr->GetCalRJoyX() );
            pc.printf("RJoyY=%2u (%6.3f)\r\n", ctrlr->GetRJoyY(), ctrlr->GetCalRJoyY() );
            if( ctrlr->IsCalibrating() ) {
                pc.puts( "Wiggle joysticks and triggers. Press HOME button to end calibration." );
            }
            else {
                pc.puts( "Press A and B buttons simultaneously to start calibration." );
            }
            
        }

        // Update counter.
        // Display is updated every so often.
        counter++;
        counter %= 10;
            
    }
    else {
        pc.puts( "READ FAILURE\r\n" );
    }
}

/** Clear the screen.
 */
static void ClearScreen( void ) {
    pc.printf( "%c[2J", 27 );                                // VT100 clear screen
}

/** Main program.
 */
int main() {
    WiiClassicControllerWithCalibration ctrlrA( PTE0, PTE1 );
    pc.baud( 115200 );
    pc.format( 8, Serial::None, 1 );
    ClearScreen();
    while (true) {
        ReadAndReport( &ctrlrA, "PORT A" );
        // If both A and B buttons are pressed and not already calibrating
        // then start calibrating and clear the screen.
        if( ctrlrA.GetButtonA() && ctrlrA.GetButtonB() && ! ctrlrA.IsCalibrating() ) {
            ctrlrA.StartCalibrating();
            ClearScreen();
        }
        // If HOME buttons is pressed and doing a calibration
        // then stop calibrating and clear the screen.
        if( ctrlrA.GetButtonHome() && ctrlrA.IsCalibrating() ) {
            ctrlrA.StopCalibrating();
            ClearScreen();
        }
        wait(LOOP_DELAY);
    }
    // Never gets here.
    // return EXIT_SUCCESS;
}
