/*
 * SOURCE FILE : WiiClassicControllerWithCalibration.h
 *
 * Definition of class WiiClassicControllerWithCalibration.
 *
 */

#ifndef WiiClassicControllerWithCalibrationDefined

    #define WiiClassicControllerWithCalibrationDefined

    #include "WiiClassicController.h"
    
    /// Derived from WiiClassicController but with calibrated analogue inputs.
    class WiiClassicControllerWithCalibration : public WiiClassicController {

    public :

        /// Enumeration of all the analogue inputs on the Wii classic controller.
        enum AnaIn {
            LeftJoyX,
            LeftJoyY,
            RightJoyX,
            RightJoyY,
            LeftTrigger,
            RightTrigger,
            AnaInCount          // MUST COME LAST!
        };
                
        /** Constructor
         * @param sda pin to use for SDA.
         * @param scl pin to use for SCL.
         */
        WiiClassicControllerWithCalibration( PinName sda, PinName scl );

        /** Destructor
         */
        virtual ~WiiClassicControllerWithCalibration();

        /** Read from the controller.
         * This override will also update calibration information
         * when calibration is in progress.
         * @returns true on success, false on failure.
         */
        virtual bool Read( void );

        /** Set scaling for a particular analogue input.
         * @param input channel to change.
         * @param m scale (multiplier) for this analogue input.
         * @param c offset for this analogue input.
         */
         void SetScaling( AnaIn input, float m, float c );
         
        /** Get scaling for a particular analogue input.
         * @param input channel to read.
         * @param m scale (multiplier) for this analogue input return here.
         * @param c offset for this analogue input returned here.
         */
         void GetScaling( AnaIn input, float *m, float *c ) const;

        /** Get calibrated left joystick X reading.
         * @returns a reading between -1 and +1.
         */
        float GetCalLJoyX( void ) const;
        
        /** Get calibrated left joystick Y reading.
         * @returns a reading between -1 and +1.
         */
        float GetCalLJoyY( void ) const;
        
        /** Get calibrated right joystick X reading.
         * @returns a reading between -1 and +1.
         */
        float GetCalRJoyX( void ) const;
        
        /** Get calibrated right joystick Y reading.
         * @returns a reading between -1 and +1.
         */
        float GetCalRJoyY( void ) const;

        /** Get calibrated left trigger reading.
         * @returns a reading between 0 and +1.
         */
        float GetCalLeftTrigger( void ) const;
        
        /** Get calibrated right trigger reading.
         * @returns a reading between 0 and +1.
         */
        float GetCalRightTrigger( void ) const;

        /** Get indication that calibration is in progress.
         * @returns true if calibration is in progress.
         */
        bool IsCalibrating( void ) const {
            return calibrating;
        }
        
        /** Start calibrating.
         * Every call to Read method updates calibration until StopCalibrating is called.
         */
        void StartCalibrating( void );
        
        /** Stop calibrating.
         */
        void StopCalibrating( void );
        
    private :

        // Indicates calibration is in progress.
        bool calibrating;
        
        // Record for each analogue input.
        class AnaInRec {
        
        public :
        
            float Scale;
            float Offset;
            UInt8 MaxCount;
            UInt8 MinCount;
            UInt8 ZeroCount;
            
            /** Calculate scale and offset for a bipolar reading.
             * @param minValue minimum value to read.
             * @param maxValue maximum value to read.
             */
             void CalculateScaleAndOffsetBiPolar( float minValue, float maxValue );
             
            /** Calculate scale and offset for a unipolar reading.
             * Minimum value is assumed to be zero.
             * @param maxValue maximum value to read.
             */
             void CalculateScaleAndOffsetUniPolar( float maxValue );
             
        };
            
        // Records for all analogue inputs.
        AnaInRec records[ (int)AnaInCount ];
        
        /** Get scaled reading.
         * @param input analogue input to scale.
         * @param raw raw readings in counts.
         * @param min minimum permitted value.
         * @param max maximum permited value.
         * @returns scaled reading.
         */
        float GetScaled( AnaIn input, UInt8 raw, float min, float max ) const;
        
        /** Update calibration information.
         */
        void UpdateCalibration( void );
        
        /** Get the raw counts for one of the analogue inputs.
         * @param input analogue input to read.
         * @returns raw counts for input.
         */
        UInt8 GetAnaIn( AnaIn input ) const;

    };

#endif

/* END of WiiClassicControllerWithCalibration.h */

