/*
 * SOURCE FILE : WiiClassicController.h
 *
 * Definition of class WiiClassicController.
 * Allows use of a Wii classic controller using an I2C bus.
 *
 */

#ifndef WiiClassicControllerDefined

  #define WiiClassicControllerDefined

  #include <mbed.h>
  #include "Types.h"
  
  #define CONTROLLER_ADDR     0xA4  // I2C library doesn't right shift the address, so provided shifted
  #define CONTROLLER_REGADDR  0x40  // relevant register address
  #define CONTROLLER_READLEN  0x06  // always read this many bytes back

  /// Allows use of a Wii classic controller using an I2C bus.
  class WiiClassicController {

  public :

    /** Constructor.
     * @param sda pin to use for SDA.
     * @param scl pin to use for SCL.
     */
    WiiClassicController( PinName sda, PinName scl );

    /** Destructor.
     */
    virtual ~WiiClassicController();

    /** Read from the controller.
     * @returns true on success, false on failure.
     */
    virtual bool Read( void );
    
    /** Read left joystick X axis.
     * @returns joystick reading as number between 0 and 63.
     */
    UInt8 GetLJoyX( void ) const {
        return (UInt8)( readBuf[ 0 ] & 0x3F );
    }
    
    /** Read left joystick Y axis.
     * @returns joystick reading as number between 0 and 63.
     */
    UInt8 GetLJoyY( void ) const {
        return (UInt8)( readBuf[ 1 ] & 0x3F );
    }
    
    /** Read right joystick X axis.
     * @returns joystick reading as number between 0 and 31.
     */
    UInt8 GetRJoyX( void ) const {
        return (UInt8)(
            ( ( readBuf[ 2 ] & 0x80 ) >> 7 ) |
            ( ( readBuf[ 1 ] & 0xC0 ) >> 5 ) |
            ( ( readBuf[ 0 ] & 0xC0 ) >> 3 )
        );
    }
    
    /** Read right joystick Y axis.
     * @returns joystick reading as number between 0 and 31.
     */
    UInt8 GetRJoyY( void ) const {
        return (UInt8)( readBuf[ 2 ] & 0x1F );
    }

    /** Read X button.
     * @returns true if button is pressed.
     */    
    bool GetButtonX( void ) const {
        return ( readBuf[ 5 ] & 0x08 ) ? false : true;
    }
    
    /** Read Y button.
     * @returns true if button is pressed.
     */    
    bool GetButtonY( void ) const {
        return ( readBuf[ 5 ] & 0x20 ) ? false : true;
    }
    
    /** Read A button.
     * @returns true if button is pressed.
     */    
    bool GetButtonA( void ) const {
        return ( readBuf[ 5 ] & 0x10 ) ? false : true;
    }
    
    /** Read B button.
     * @returns true if button is pressed.
     */    
    bool GetButtonB( void ) const {
        return ( readBuf[ 5 ] & 0x40 ) ? false : true;
    }
    
    /** Read left trigger button.
     * @returns true if button is pressed.
     */    
    bool GetButtonLT( void ) const {
        return ( readBuf[ 4 ] & 0x20 ) ? false : true;
    }
    
    /** Read right trigger button.
     * @returns true if button is pressed.
     */    
    bool GetButtonRT( void ) const {
        return ( readBuf[ 4 ] & 0x02 ) ? false : true;
    }
    
    /** Read left trigger reading.
     * @returns reading as a number between 0 and 31.
     */    
    UInt8 GetLeftTrigger( void ) const {
        return (UInt8)(
            ( ( readBuf[ 3 ] & 0xE0 ) >> 5 ) |
            ( ( readBuf[ 2 ] & 0x60 ) >> 2 )
        );
    }
    
    /** Read right trigger reading.
     * @returns reading as a number between 0 and 31.
     */    
    UInt8 GetRightTrigger( void ) const {
        return (UInt8)( readBuf[ 3 ] & 0x1F );
    }
    
    /** Read ZL button.
     * @returns true if button is pressed.
     */    
    bool GetButtonZL( void ) const {
        return ( readBuf[ 5 ] & 0x80 ) ? false : true;
    }
    
    /** Read ZR button.
     * @returns true if button is pressed.
     */    
    bool GetButtonZR( void ) const {
        return ( readBuf[ 5 ] & 0x04 ) ? false : true;
    }
    
    /** Read select (or minus) button.
     * @returns true if button is pressed.
     */    
    bool GetButtonSelect( void ) const {
        return ( readBuf[ 4 ] & 0x10 ) ? false : true;
    }

    /** Read home button.
     * @returns true if button is pressed.
     */    
    bool GetButtonHome( void ) const {
        return ( readBuf[ 4 ] & 0x08 ) ? false : true;
    }
    
    /** Read start (or plus) button.
     * @returns true if button is pressed.
     */    
    bool GetButtonStart( void ) const {
        return ( readBuf[ 4 ] & 0x04 ) ? false : true;
    }
    
    /** Read up button.
     * @returns true if button is pressed.
     */    
    bool GetButtonUp( void ) const {
        return ( readBuf[ 5 ] & 0x01 ) ? false : true;
    }
    
    /** Read down button.
     * @returns true if button is pressed.
     */    
    bool GetButtonDown( void ) const {
        return ( readBuf[ 4 ] & 0x40 ) ? false : true;
    }
    
    /** Read left button.
     * @returns true if button is pressed.
     */    
    bool GetButtonLeft( void ) const {
        return ( readBuf[ 5 ] & 0x02 ) ? false : true;
    }
    
    /** Read right button.
     * @returns true if button is pressed.
     */    
    bool GetButtonRight( void ) const {
        return ( readBuf[ 4 ] & 0x80 ) ? false : true;
    }
    
    /** Get size of read buffer.
     * @returns size of read buffer.
     */
    UInt8 GetReadBufSize( void ) const {
        return sizeof(readBuf);
    }
    
    /** Get address of read buffer.
     * @returns pointer to read buffer.
     */
    UInt8* GetReadBuf( void ) {
        return readBuf;
    }

  private :

    // I2C port used for comms with controller.  
    I2C controllerPort;
    
    // Set to true once controller has been initialised.
    bool initialised;
    
    // Buffer for incoming data.
    UInt8 readBuf[ CONTROLLER_READLEN ];

    /** Initialise the controller.
     * @returns true on success, false on failure.
     */
    bool ControllerInit( void );
    
    /** Read from the controller, assuming it has been initialised.
     * @returns true on success, false on failure.
     */
    bool ControllerRead( void );

    /** Decoder a byte in received packet.
     * @param x byte to decode
     * @returns decoded byte
     */
    UInt8 Decode( UInt8 x ) {
        return (x ^ 0x17) + 0x17;
    }
    
  };

#endif

/* END of WiiClassicController.h */


