/*
 * SOURCE FILE : Walker.cpp
 *
 * Methods for objects that walk sideways on (like humans and crushers).
 * 
 */

#include "Gameduino.h"                    // Gameduino stuff
#include "GDConst.h"
#include "Walker.h"
#include "Random.h"

/*************************************************/
/* INITIALISE HORIZONTAL AND VERTICAL VELOCITIES */
/*************************************************/
// Pass pointers to horizontal and vertical velocities in hv and vv.
void Walker::InitialiseVelocities( Int16 *hv, Int16 *vv ) {
  // Initialise horizontal and vertical speeds.
  UInt8 rnd = Random::Get( 4 );
  *hv = GameObject::FromPixel( 1 ) >> 2;
  if( rnd & 1 ) {
    *hv = -(*hv);
  }
  *vv = GameObject::FromPixel( 1 ) >> 4;
  if( rnd & 2 ) {
    *vv = -(*vv);
  }
}

/*********************************************************/
/* UPDATE COORDINATES AND VELOCITIES TO MAKE OBJECT WALK */
/*********************************************************/
// Pass pointers to x and y coordinates in x and y.
// Pass pointers to horizontal and vertical velocities in hv and vv.
// Pass restriction flags (as defined in GameObject.h) in restriction Flags.
void Walker::Walk( Int16 *x, Int16 *y, Int16 *hv, Int16 *vv, UInt8 restrictionFlags ) {
    // Make object bounce of edges of restriction area.
    if( restrictionFlags & (UInt8)( GameObject::LeftRestriction | GameObject::RightRestriction ) ) {
      *hv = -(*hv);
    }
    if( restrictionFlags & (UInt8)( GameObject::UpRestriction | GameObject::DownRestriction ) ) {
      *vv = -(*vv);
    }
    *x += *hv;
    *y += *vv;
}

/*************************/
/* DRAW A WALKING OBJECT */
/*************************/
// Pass pointer to Gameduino to draw on in gd.
// Pass sprite number in spriteNumber.
// Pass x and y coordinates in x and y (NOT pixel coordinates).
// Pass counter used to pace animation in frameCounter.
// Pass pointer to animation data (array of AnimationStages sprite image numbers) in animationData.
void Walker::Draw( Gameduino *gd, UInt8 spriteNumber, Int16 x, Int16 y, Int16 hv, UInt8 frameCounter, const UInt8 *animationData ) {
    // Reverse sprite image horizontally if horizontal speed is positive.
    Gameduino::Rotation transform = ( hv < 0 ) ? Gameduino::None : Gameduino::FlipX;
    // Image number changes with the frame counter.
    const UInt8 *address = animationData + frameCounter % AnimationStages;
    // Update sprite location and image. Note the last parameter is BadGuy so that it is possible
    // for player to collide with a human. GoodGuy would mean collisions would be ignored.
    gd->sprite( spriteNumber, GameObject::ToPixel( x ), GameObject::ToPixel( y ), *address, 0, transform, BadGuy );
}
