/*
 * SOURCE FILE : GameObject.h
 *
 * The abstract base class for all graphical game objects.
 *
 */

#ifndef GameObjectIncluded
  
  #define GameObjectIncluded

  #include "Gameduino.h"              // Gameduino stuff
  #include "Types.h"
  #include "GDConst.h"
  #include "SpriteImageId.h"
  #include "SpriteGroup.h"
  #include "GameObjectTypes.h"
  #include "Rectangle.h"
  
  class GameObject {
    
  public :

    // Bitmasks to use with RestrictionFlags.
    enum RestrictionMask {
      UpRestriction = 1,
      DownRestriction = 2,
      LeftRestriction = 4,
      RightRestriction = 8,
    };
    
    // X coordinate (NOT pixel coordinate).
    Int16 Xco;
    
    // Y coordinate (NOT pixel coordinate).
    Int16 Yco;

    // Pixel width.
    UInt8 PixelWidth;
    
    // Pixel height.
    UInt8 PixelHeight;
    
    // Visibility flag.
    bool Visible;
    
    // Limits for object movement.
    Rectangle *Bounds;
    
    // Indicates if movement is restricted.
    bool MovementRestricted;
    
    // Indicates object should be deleted when moves outside restricted area.
    bool DeleteWhenRestricted;
    
    // Flags which are set when object is being restricted.
    // Only works when MovementRestricted is true.
    // If MovementRestricted is false RestrictionFlags are always all zero.
    // Use RestrictionMask enumeration with this field.
    UInt8 RestrictionFlags;
    
    // Indicates which Gameduino sprite is being used for this object.
    UInt8 SpriteNumber;
    
    // Determines if object is retained on a level restart.
    // Most enemy objects are retained on a level restart, although their positions change.
    // However, enemy bullets (for example) are killed off on a level restart.
    bool RetainOnLevelRestart;
        
    /***************/
    /* CONSTRUCTOR */
    /***************/
    GameObject() :
      Xco( FromPixel( 100 ) ),
      Yco( FromPixel( 100 ) ),
      PixelWidth( SPRITE_PIXEL_WIDTH ),
      PixelHeight( SPRITE_PIXEL_HEIGHT ),
      Visible( true ),
      Bounds( NULL ),
      MovementRestricted( true ),
      DeleteWhenRestricted( false ),
      RestrictionFlags( 0 ),
      SpriteNumber( 0 ),
      RetainOnLevelRestart( true )
    {
    }

    /**************/
    /* DESTRUCTOR */
    /**************/
    // Must be virtual!
    // If not then could end up calling the base class
    // destructor for a derived class. Not pretty.
    virtual ~GameObject() {
    }
     
    /************************/
    /* GET GAME OBJECT TYPE */
    /************************/
    // Returns type of game object.
    virtual GameObjectTypes GetType( void ) = 0;
    
    // Note that coordinates for all game objects are NOT screen pixel coordinates.
    // The coordinate system used has a much higher resolution that the screen.
    // This allows objects to move at a speed of less than one pixel per screen update.
    // Use the methods ToPixel and FromPixel to convert between game object coordinates
    // and pixel coordinates.
    
    /********************************/
    /* CONVERT TO PIXEL COORDINATES */
    /********************************/
    // Returns pixel equivalent of coordinate.
    static Int16 ToPixel( Int16 x ) {
      return ( x >> 6 );
    }
    
    /**********************************/
    /* CONVERT FROM PIXEL COORDINATES */
    /**********************************/
    // Returns pixel equivalent of coordinate.
    static Int16 FromPixel( Int16 x ) {
      return ( x << 6 );
    }
    
    /************************/
    /* MOVE THE GAME OBJECT */
    /************************/
    void Move( void ) {
      ProtectedMove();
      // Calculate coordinates for edges of object.
      Int16 dx = FromPixel( ( SPRITE_PIXEL_WIDTH - PixelWidth ) >> 1 );
      Int16 dy = FromPixel( ( SPRITE_PIXEL_HEIGHT - PixelHeight ) >> 1 );
      Int16 left = Xco + dx;
      Int16 top = Yco + dy;
      Int16 right = left + FromPixel( PixelWidth - 1 );
      Int16 bottom = top + FromPixel( PixelHeight - 1 );
      RestrictionFlags = 0;
      // Kill off or restrict movement if moved outside restricted area.
      // Do nothing if Bounds is NULL.
      if( Bounds != NULL ) {
        if( DeleteWhenRestricted ) {
          if(
            left < Bounds->X1 ||
            right > Bounds->X2 ||
            top < Bounds->Y1 ||
            bottom > Bounds->Y2
          ) {
            Visible = false;
          }
        }
        else if( MovementRestricted ) {
          if( left < Bounds->X1 ) {
            Xco = Bounds->X1 - dx;
            RestrictionFlags |= (UInt8)LeftRestriction;
          }
          else if( right > Bounds->X2 ) {
            Xco = Bounds->X2 - FromPixel( PixelWidth - 1 );
            RestrictionFlags |= (UInt8)RightRestriction;
          }
          if( top < Bounds->Y1 ) {
            Yco = Bounds->Y1 - dy;
            RestrictionFlags |= (UInt8)UpRestriction;
          }
          else if( bottom > Bounds->Y2 ) {
            Yco = Bounds->Y2 - FromPixel( PixelHeight - 1 );
            RestrictionFlags |= (UInt8)DownRestriction;
          }
        }
      }
    }

    /************************/
    /* DRAW THE GAME OBJECT */
    /************************/
    // Pass pointer to Gameduino to draw on in gd.
    // Note if Visible is false this should not draw anything
    // and/or hide the visible object.
    virtual void Draw( Gameduino *gd ) = 0;
    
    /**********************************/
    /* INITIALISE AN ARRAY OF OBJECTS */
    /**********************************/
    // Really only intended for the initialisation of enemy objects and humans.
    // Each object in the array is allocated a consecutive sprite number and is positioned
    // randomly in the arena. The objects movement is restricted to within the arena.
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
    // Pass pointer to a sprite number in spriteNumber. This number is incremented by this method.
    static void InitialiseAll( GameObject **objects, UInt8 objectCount, UInt8 *spriteNumber );

    /****************************/
    /* MOVE AN ARRAY OF OBJECTS */
    /****************************/
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
    // Returns true if any non-null objects were found in the array.
    static bool MoveAll( GameObject **objects, UInt8 objectCount );

    /****************************/
    /* DRAW AN ARRAY OF OBJECTS */
    /****************************/
    // Pass pointer to Gameduino to draw on in gd.
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
    static void DrawAll( Gameduino *gd, GameObject **objects, UInt8 objectCount );

    /************************************************/
    /* FIND AN UNUSED OBJECT IN AN ARRAY OF OBJECTS */
    /************************************************/
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
    // Pass pointer to variable that will hold index of object found in index.
    // Returns true if an unused object was found, false if not.
    // An unused object is indicated by a null pointer in the array.
    static bool FindUnusedObject( GameObject **objects, UInt8 objectCount, UInt8 *index );

    /****************************************************/
    /* FIND COLLISIONS WITH ALL THE OBJECTS IN AN ARRAY */
    /****************************************************/
    // Pass pointer to Gameduino in gd parameter.
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
    // Pass pointer to a function that takes two UInt8 parameters in func.
    // The first parameter is the index of the object in the objects array that hit something.
    // The second parameter is the sprite number of the sprite which it hit.
    static void FindCollisions( Gameduino *gd, GameObject **objects, UInt8 objectCount, void (*func)( UInt8, UInt8 ) );
    
    /*************************************************************************/
    /* FIND AN OBJECT WITH A PARTICULAR SPRITE NUMBER IN AN ARRAY OF OBJECTS */
    /*************************************************************************/
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
    // Pass sprite number to look for in spriteNumber.
    // Index of object with given sprite number written to variable pointed to by index.
    // Returns true if sprite number was found, false if not.
    static bool FindSpriteNumber( GameObject **objects, UInt8 objectCount, UInt8 spriteNumber, UInt8 *index );

    /**********************************************/
    /* FIND NEAREST OBJECT IN AN ARRAY OF OBJECTS */
    /**********************************************/
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
        // Pass x and y coordinates of point you want to check.
        // Pass pointer to validation function in ValidFunc.
        // This is used to establish if a particular object is to be considered
        // when finding nearest object. It should return true if object should be considered
        // or false to ignore it. Pass NULL if all objects are considered valid.
    // Pass pointer to variable that will hold index of object found in index.
    // Returns true if nearest object was found, false if not (maybe no objects in array).
    static bool FindNearestObject(
            GameObject **objects, UInt8 objectCount,
            Int16 x, Int16 y,
            bool (*ValidFunc)( GameObject *object ),
            UInt8 *index
        );

        /***************************************************************************/
        /* REMOVE ALL OBJECTS IN AN ARRAY THAT ARE NOT RETAINED ON A LEVEL RESTART */
        /***************************************************************************/
    // Pass pointer to array of pointers to GameObjects in objects.
    // Pass number of pointers in the array in objectCount.
        // All objects pointed to in the array that have their RetainOnLevelRestart property set
        // to false are removed by writing NULL into the array.
        static void RemoveUnretainedObjects( GameObject **objects, UInt8 objectCount );
        
        /*******************************/
        /* MOVE TOWARDS ANOTHER OBJECT */
        /*******************************/
        // Pass object to move towards in object.
        // Pass speed at which to move in speed.
        void MoveTowards( GameObject *object, Int16 speed );
        
  protected :
  
    /************************/
    /* MOVE THE GAME OBJECT */
    /************************/
    virtual void ProtectedMove( void ) = 0;
  
  private :

  };
    
#endif

/* END of GameObject.h */


