/*
 * SOURCE FILE : HighScoreTable.h
 *
 * Definition of class HighScoreTable.
 * Maintains a table of high scores with a name and a score for each entry in the table.
 *
 */

#ifndef HighScoreTableDefined

  #define HighScoreTableDefined

  #include "Types.h"
  #include "PlayerName.h"
  #include "Ser25lcxxx.h"       // serial EEPROM routines. Also work for 25AAxxx EEPROMs.
  
  class HighScoreTable {

  public :

    /***************/
    /* CONSTRUCTOR */
    /***************/
    // Pass pointer to an SPI EEPROM which contains the high scores.
    HighScoreTable( Ser25LCxxx *e );

    /**************/
    /* DESTRUCTOR */
    /**************/
    virtual ~HighScoreTable();

    /***************************************/
    /* GET EEPROM USED BY HIGH SCORE TABLE */
    /***************************************/
    Ser25LCxxx *GetEEPROM( void ) const {
        return eeprom;
    }
        
    /****************************************/
    /* VALIDATE EEPROM USED FOR HIGH SCORES */
    /****************************************/
    // Checks EEPROM used for high scores and
    // if any of it looks like nonsense it
    // rewrites the whole table with defaults.
    void ValidateEEPROM( void );

    /********************************/
    /* DETERMINE IF EEPROM IS VALID */
    /********************************/
    // Returns true if EEPROM is valid.
    bool EEPROMValid( void );
    
    /****************************/
    /* WRITE DEFAULTS TO EEPROM */
    /****************************/
    void WriteEEPROMDefaults( void );
    
    /*************************/
    /* GET CAPACITY OF TABLE */
    /*************************/
    // Returns capacity of table.
    UInt8 GetCapacity( void ) const {
      return capacity;
    }
    
    /**********************************************/
    /* DETERMINE POSITION OF A SCORE IN THE TABLE */
    /**********************************************/
    // Pass score in score.
    // Returns position in table (0 is top score).
    // If position returned is >= capacity of table then score is not high
    // enough to place in table.
    UInt8 GetPositionInTable( UInt32 score ) const;
    
    /*********************************/
    /* ADD ENTRY TO HIGH SCORE TABLE */
    /*********************************/
    // Pass position in table to put entry in pos.
    // Pass name of player in name.
    // Pass score in score.
    void Add( UInt8 pos, const PlayerName *name, UInt32 score );

    /****************************/
    /* GET ENTRY FROM THE TABLE */
    /****************************/
    // Pass position to fetch from in pos.
    // Player name is returned in object pointed to by name.
    // Player score is returned in integer pointed to by score.
    void Get( UInt8 pos, PlayerName *name, UInt32 *score ) const;
    
  private :

    // Change the following to suit.
    enum {
        eepromAddress = 0,    // address in EEPROM where high score table starts
        capacity = 10,        // number of entries in table
        // Amount of memory used in EEPROM.
        // You need capacity bytes for the index and then
        // eight bytes for each high score and name.
        memoryUsed = capacity + ( capacity << 3 ),
    };

    // Pointer to EEPROM that holds high scores.
    Ser25LCxxx *eeprom;
        
  };

#endif

/* END of HighScoreTable.h */
