/*
 * SOURCE FILE : OneShotObject.h
 *
 * Base class for all objects that do not move and play an animation once before vanishing.
 * Useful for explosions and popup scores.
 *
 */

#ifndef OneShotObjectIncluded
  
  #define OneShotObjectIncluded

  #include "GameObject.h"
  #include "SpriteImageId.h"
  
  class OneShotObject : public GameObject {
    
  public :

    /***************/
    /* CONSTRUCTOR */
    /***************/
    OneShotObject() :
      imageNumber( Explosion0Image ),
      firstImageNumber( Explosion0Image ),
      lastImageNumber( Explosion9Image ),
      maxCountdown( 7 ),
      imageCountdown( 7 )
    {
      MovementRestricted = false;
      DeleteWhenRestricted = false;
    }

    /**************/
    /* DESTRUCTOR */
    /**************/
    virtual ~OneShotObject() {
    }
    
    /**************************************************************/
    /* SET RANGE OF SPRITE IMAGES TO USE AND DELAY FOR EACH IMAGE */
    /**************************************************************/
    // Pass first sprite image used in first.
    // Pass last sprite image used in last.
    // Pass delay count between images in count.
    void SetImageRange( UInt8 first, UInt8 last, UInt8 count ) {
      firstImageNumber = first;
      lastImageNumber = last;
      maxCountdown = count;
      Restart();
    }
    
    /************************/
    /* MOVE THE GAME OBJECT */
    /************************/
    virtual void ProtectedMove( void );

    /************************/
    /* DRAW THE GAME OBJECT */
    /************************/
    // This is only called after it has been established that the
    // game object is visible.
    virtual void Draw( Gameduino *gd ) {
      gd->sprite( SpriteNumber, ToPixel( Xco ), ToPixel( Yco ), imageNumber, 0, Gameduino::None, GoodGuy );
    }

    /****************************/
    /* RESTART FROM FIRST IMAGE */
    /****************************/
    void Restart( void ) {
      imageNumber = firstImageNumber;
      imageCountdown = maxCountdown;
      Visible = true;
    }
    
  private :
  
    // Image number currently in use.
    UInt8 imageNumber;
    
    // Image numbers of first and last images.
    UInt8 firstImageNumber, lastImageNumber;
    
    // Maximum countdown to next image change.
    // Determines speed of explosion animation.
    UInt8 maxCountdown;
    
    // Countdown to next image.
    UInt8 imageCountdown;

  };
    
#endif

/* END of OneShotObject.h */

