/*
 * SOURCE FILE : GDExtra.cpp
 *
 * Additional routines for the Gameduino.
 *
 */

#include "GDExtra.h"
#include "GDConst.h"

/*********************************************/
/* DISPLAY A STRING STORED IN PROGRAM MEMORY */
/*********************************************/
// Pass pointer to a Gameduino in gd.
// Pass coordinates in x and y.
// Pass address of string in s.
void GDExtra::WriteProgString( Gameduino *gd, UInt8 x, UInt8 y, const char *s ) {
  UInt16 address = Gameduino::RAM_PIC + y * SCREEN_CHAR_WIDTH + x;
  while( *s != 0 ) {
    gd->wr( address++, *s++ );
  }
}

/*******************************/
/* DISPLAY AN UNSIGNED INTEGER */
/*******************************/
// Pass pointer to a Gameduino in gd.
// Pass coordinates in x and y.
// Pass number to write in num.
// Pass numeric base in base.
// Pass number of digits to write in digits.
void GDExtra::WriteUInt16( Gameduino *gd, UInt8 x, UInt8 y, UInt16 num, UInt8 base, UInt8 digits ) {
  UInt16 address = Gameduino::RAM_PIC + y * SCREEN_CHAR_WIDTH + x + digits - 1;
  UInt8 dig;
  while( digits > 0 ) {
    dig = num % base;
    if( dig < 10 ) {
      gd->wr( address--, '0' + dig );
    }
    else {
      gd->wr( address--, 'A' + dig );
    }
    num /= base;
    digits--;
  }
}

/************************/
/* DISPLAY A BCD NUMBER */
/************************/
// Pass pointer to a Gameduino in gd.
// Pass coordinates in x and y.
// Pass number to write in num.
// Pass number of digits to write in digits.
void GDExtra::WriteBCDNumber( Gameduino *gd, UInt8 x, UInt8 y, UInt32 num, UInt8 digits ) {
  UInt16 address = Gameduino::RAM_PIC + y * SCREEN_CHAR_WIDTH + x + digits - 1;
  while( digits > 0 ) {
    gd->wr( address--, '0' + ( num & 0xF ) );
    num >>= 4;
    digits--;
  }
}

/******************************************************/
/* DISPLAY A CHARACTER BLOCK STORED IN PROGRAM MEMORY */
/******************************************************/
// Pass pointer to a Gameduino in gd.
// Pass coordinates in x and y.
// Pass address of character block in bPtr.
// Address points to the width of the block, followed by the height of
// the block and then the character codes that make up the block.
void GDExtra::WriteProgCharBlock( Gameduino *gd, UInt8 x, UInt8 y, const UInt8 *bPtr ) {
  // Calculate address for top left of block.
  UInt16 address = Gameduino::RAM_PIC + y * SCREEN_CHAR_WIDTH + x;
  // Fetch width and height from program memory.
  UInt8 width = *bPtr++;
  UInt8 height = *bPtr++;
  // Write rows of data to screen.
  for( int row = 0; row < height; ++row ) {
    gd->copy( address, (prog_uchar*)bPtr, width );
    address += SCREEN_CHAR_WIDTH;
    bPtr += width;
  }
}

/****************/
/* CLEAR SCREEN */
/****************/
// Pass pointer to a Gameduino in gd.
// Writes all character RAM with given character code.
void GDExtra::ClearScreen( Gameduino *gd, UInt8 code ) {
  gd->fill( Gameduino::RAM_PIC, code, RAM_PIC_SIZE );
}

/************************/
/* HIDE A SINGLE SPRITE */
/************************/
// Pass pointer to a Gameduino in gd.
// Pass sprite number in spriteNumber parameter.
void GDExtra::HideSprite( Gameduino *gd, UInt8 spriteNumber ) {
    gd->sprite( spriteNumber, 0, 400, 0, 0 );
}

/********************/
/* HIDE ALL SPRITES */
/********************/
// Pass pointer to a Gameduino in gd.
void GDExtra::HideAllSprites( Gameduino *gd ) {
  // This code comes from the Cookbook section of the Gameduino
  // website. Look for the entry entitled "Hiding sprites".
  gd->__wstartspr(0);  // start writing sprites at slot 0
  // Now hide all the remaining sprites.
  // Actually this does not hide sprite number 255.
  // If you used SPRITE_COUNT instead of SPRITE_COUNT - 1 then
  // you end up with an endless loop because gd->spr is a byte
  // and is always less than SPRITE_COUNT since SPRITE_COUNT is 256.
  while( gd->spr < SPRITE_COUNT - 1 ) {
    gd->xhide();
  }
  gd->__end();
}
