/*
 * SOURCE FILE : SoundManager.h
 *
 * Responsible for playing simple sounds.
 *
 */

#ifndef SoundManagerIncluded
  
  #define SoundManagerIncluded

  #include <stdio.h>           // for NULL
  #include "Types.h"           // various types
  #include "Gameduino.h"       // Gameduino stuff
  
  class SoundManager {
    
  public :

    // An instance of this class.
    static SoundManager Instance;
    
    /***************/
    /* CONSTRUCTOR */
    /***************/
    SoundManager();

    /*******************************************/
    /* SET GAMEDUINO THAT WILL PLAY THE SOUNDS */
    /*******************************************/
    // Pass pointer to a Gameduino which will play the sounds.
    // You must call this before using other routines.
    void SetGameduino( Gameduino *gameduino ) {
        gd = gameduino;
    }

    /************************/
    /* TRY AND PLAY A SOUND */
    /************************/
    // Pass address of tune in program memory in tuneAddress.
    // Pass bias to apply to left speaker amplitude in leftAmpBias.
    // Pass bias to apply to right speaker amplitude in leftAmpBias.
    // Returns true if sound was started, false if no sound slots free.
    bool PlaySound( const UInt8 *tuneAddress, UInt8 leftAmpBias, UInt8 rightAmpBias );
    
    /****************************/
    /* UPDATE PLAYING OF SOUNDS */
    /****************************/
    // Should be called at regular intervals.
    void Update( void );

    /**********************************/
    /* COUNT NUMBER OF SOUNDS PLAYING */
    /**********************************/
    // Returns number of sounds playing.
    UInt8 CountSoundsPlaying( void );
    
    /******************************/
    /* SILENCE ALL SOUNDS PLAYING */
    /******************************/
    void SilenceAll( void );
    
  private :

    enum {
      FirstVoice = 0,        // First Gameduino voice channel used
      VoiceCount = 8,        // Number of voice channels used
    };

    // Record for tune associated with each voice used.
    class TuneRecord {
    public :
    
      TuneRecord() :
        TuneAddress( (UInt8*)NULL )
      {
      }
      
      Gameduino::WaveForm TuneWaveform;
      UInt8 LeftAmpBias, RightAmpBias;
      const UInt8 *TuneAddress;
      UInt8 Countdown;
      
    };
    
    // Records used to step through each tune.
    // A value of zero in TuneAddress field indicates an unused slot.
    TuneRecord tuneRecords[ VoiceCount ];

    // Gameduino that plays the sounds.
    Gameduino *gd;
        
    /*****************************/
    /* UPDATE A SINGLE TUNE SLOT */
    /*****************************/
    // Do NOT call with a record containing zero in TuneAddress (a free slot).
    void UpdateTuneSection( UInt8 voiceNum, TuneRecord *rec );
    
  };
  
#endif

/* END of SoundManager.h */

