/*
 * SOURCE FILE : EnemyObject.h
 *
 * Base class for enemy objects.
 *
 */

#ifndef EnemyObjectIncluded
  
  #define EnemyObjectIncluded

  #include "GameObject.h"
  #include "PlayerObject.h"
    #include "EnemyType.h"

  class EnemyObject : public GameObject {
    
  public :

    enum {
        Indestructable = 0xFF,
    };

    // Number of hit points remaining.
    // When this reaches zero the enemy dies.
    // However, if this has the special value Indestructable then enemy cannot be killed.
    UInt8 HitPoints;
    
    // Set to true of the enemy squashes humans.
    bool SquashesHumans;
    
    // Pointer to array of pointers to enemies.
    // Some enemies may need to add new enemies (such as bullets or new spawning enemies) to this array.
    GameObject **Enemies;
        
    /***************/
    /* CONSTRUCTOR */
    /***************/
    EnemyObject() :
            HitPoints( 1 ),
            SquashesHumans( false ),
            Enemies( (GameObject**)NULL ),
      chaseObject( &defaultChaseObject )
    {
    }

    /**************/
    /* DESTRUCTOR */
    /**************/
    virtual ~EnemyObject() {
    }
    
    /************************/
    /* GET GAME OBJECT TYPE */
    /************************/
    // Returns type of game object.
    virtual GameObjectTypes GetType( void ) {
      return EnemyObjectType;
    }

    /*****************************/
    /* GET TYPE OF ENEMY THIS IS */
    /*****************************/
    // Returns enemy type.
    virtual EnemyType GetEnemyType( void ) = 0;
        
    /*******************************************************/
    /* GET NUMBER OF POINTS AWARDED FOR KILLING THIS ENEMY */
    /*******************************************************/
    // Returns number of points (in BCD).
    virtual UInt8 GetPoints( void ) = 0;

    /***********************/
    /* SET OBJECT TO CHASE */
    /***********************/
    // Pass pointer to object to chase in co.
    void SetChaseObject( GameObject *co ) {
      chaseObject = co;
    }

    /*****************************************************************/
    /* PERFORM ANY INIITALISATION REQUIRED AT LEVEL START OR RESTART */
    /*****************************************************************/
    // Derived classes should override this if necessary.
    virtual void LevelRestart( void ) {
    }
        
    /********************************************/
    /* INFORM ENEMY IT HAS BEEN HIT BY A BULLET */
    /********************************************/
    // Default implementation does nothing but if special behaviour
    // is required in a derived class then this should be overridden.
    // Note that this does NOT deal with determining if the enemy is dead or not.
    // An enemy ALWAYS dies if HitPoints reaches zero.
    virtual void RegisterHitByBullet( void ) {}
    
    /*****************************************************/
    /* CHECK IF ALL SURVIVING ENEMIES ARE INDESTRUCTABLE */
    /*****************************************************/
    // Pass pointer to array of pointers to EnemyObjects in enemies.
    // Pass number of pointers in the array in enemyCount.
    static bool AreAllIndestructable( const EnemyObject **enemies, UInt8 enemyCount );
        
  protected :

    // Object to chase.
    GameObject *chaseObject;
    
  private :
  
    // Default object to chase.
    static PlayerObject defaultChaseObject;
    
  };
    
#endif

/* END of EnemyObject.h */


