/*
 * SOURCE FILE : BulletVelocities.h
 *
 * Converts from set of joystick inputs to horizontal and vertical velocities.
 *
 */

#ifndef BulletVelocitiesIncluded
  
  #define BulletVelocitiesIncluded

  #include "Int16Pair.h"
  
  class BulletVelocities {
  
  public :

    /***************/
    /* CONSTRUCTOR */
    /***************/
    // Pass velocity for a horizontally moving bullet in h.
    // Pass velocity for a vertically moving bullet in v.
    BulletVelocities( Int16 h, Int16 v ) {
      // Calculate velocities used on the diagonal.
      // This is really doing hd = h * cos( 45 degrees ) and
      // vd = v * sin( 45 degrees ).
      Int16 hd = (Int16)floor( (double)h * 0.707 + 0.5 );
      Int16 vd = (Int16)floor( (double)v * 0.707 + 0.5 );
      // Initialise the table of velocities.
      Int16Pair *ptr = pairs;
      // No joystick contacts closed. Index 0.
      ptr->X = 0;
      ptr->Y = 0;
      ptr++;
      // Joystick up. Index 1
      ptr->X = 0;
      ptr->Y = -v;
      ptr++;
      // Joystick down. Index 2.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick up and down (impossible). Index 3.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick left. Index 4.
      ptr->X = -h;
      ptr->Y = 0;
      ptr++;
      // Joystick left and up. Index 5.
      ptr->X = -hd;
      ptr->Y = -vd;
      ptr++;
      // Joystick left and down. Index 6.
      ptr->X = -hd;
      ptr->Y = vd;
      ptr++;
      // Joystick up, down and left (impossible). Index 7.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick right. Index 8.
      ptr->X = h;
      ptr->Y = 0;
      ptr++;
      // Joystick right and up. Index 9.
      ptr->X = hd;
      ptr->Y = -vd;
      ptr++;
      // Joystick right and down. Index 10.
      ptr->X = hd;
      ptr->Y = vd;
      ptr++;
      // Joystick up, down and right (impossible). Index 11.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick left and right (impossible). Index 12.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick up, left and right (impossible). Index 13.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick down, left and right (impossible) Index 14.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
      // Joystick up, down, left and right (impossible). Index 15.
      ptr->X = 0;
      ptr->Y = v;
      ptr++;
    }

    /*******************************************************/
    /* GET VELOCITIES FOR A COMBINATION OF JOYSTICK INPUTS */
    /*******************************************************/
    // Pass a map containing joystick inputs where a set bit indicates a closed contact.
    // Bits must be ordered as follows :
    // Bit 0 = Up.
    // Bit 1 = Down.
    // Bit 2 = Left.
    // Bit 3 = Right.
    // Remaining bits are ignored.
    const Int16Pair *GetVelocities( UInt8 joyMap ) const {
      return pairs + ( joyMap & 0x0F );
    }
    
  private :

    // Horizontal and vertical velocities for each combination of joystick inputs.    
    Int16Pair pairs[ 16 ];
    
  };
  
#endif

/* END of BulletVelocities.h */

