/*
 * SOURCE FILE : BulletManager.cpp
 *
 * Responsible for managing a collection of bullets.
 *
 */

#include "BulletManager.h"
#include "BulletObject.h"
#include "ArenaConst.h"
#include "GDExtra.h"

/***************/
/* CONSTRUCTOR */
/***************/
// Pass number of sprite to use for first bullet.
// MaxBullets consequtive sprites will be used for bullets.
BulletManager::BulletManager( UInt8 firstSpriteNumber ) {
  // Initialise aspects of bullets that never change.
  BulletObject *bullet;
  for( UInt8 b = 0; b < MaxBullets; ++b ) {
    bullet = bullets + b;
    bullet->SpriteNumber = firstSpriteNumber + b;
    bullet->PixelWidth = 6;
    bullet->PixelHeight = 6;
    bullet->Bounds = &ArenaRectangle;
    // Make sure entry in bulletPointers array is NULL.
    bulletPointers[ b ] = (GameObject*)NULL;
  }
}

/**********************/
/* START A BULLET OFF */
/**********************/
// Pass start coordinates in x and y (NOT pixel coordinates).
// Pass bullet velocities in hv and vv (NOT pixel velocities).
// Returns true if bullet was started successfully.
bool BulletManager::StartBullet( Int16 x, Int16 y, Int16 hv, Int16 vv ) {
  // Try and find an unused bullet slot.
  UInt8 index;
  if( GameObject::FindUnusedObject( bulletPointers, MaxBullets, &index ) ) {
    // Found a free bullet slot. Point to entry in bullets array.
    BulletObject *bullet = bullets + index;
    bulletPointers[ index ] = bullet;
    bullet->Start( x, y, hv, vv );
    return true;
  }
  else {
    // No free bullet slots.
    return false;
  }
}

/************************/
/* KILL A SINGLE BULLET */
/************************/
// Pass pointer to Gameduino that displays bullets in gd.
// Pass index of bullet in b.
void BulletManager::KillBullet( Gameduino *gd, UInt8 b ) {
    GameObject *bullet = bulletPointers[ b ];
    if( bullet != (GameObject*)NULL ) {
      bulletPointers[ b ] = (BulletObject*)NULL;
      // Hide bullet sprite by setting y coordinate to 400.
      GDExtra::HideSprite( gd, bullet->SpriteNumber );
    }
}

/********************/
/* KILL ALL BULLETS */
/********************/
// Pass pointer to Gameduino that displays bullets in gd.
void BulletManager::KillAllBullets( Gameduino *gd ) {
  for( UInt8 b = 0; b < MaxBullets; ++b ) {
    KillBullet( gd, b );
  }
}
