/*
 * SOURCE FILE : PanelControls.h
 *
 * Responsible for reading joysticks and buttons.
 * This is the version for mbed LPC1768.
 * There is a version kicking around for the LPC11U24 that uses different inputs for joystick 2.
 *
 */

#ifndef PanelControlsIncluded
  
  #define PanelControlsIncluded

  #include "mbed.h"
  #include "Types.h"
  
  class PanelControls {
    
  public :

    /***************/
    /* CONSTRUCTOR */
    /***************/
    PanelControls() :
      inputs( 0 ),
      bus( (BusIn*)NULL )
    {
    }
    
    /**************/
    /* DESTRUCTOR */
    /**************/
    ~PanelControls() {
      delete bus;
    }
    
    /*************************/
    /* INITIALISE INPUT PINS */
    /*************************/
    void InitialisePins( void ) {
      // Create an input bus.
      bus = new BusIn(
        p23, p24, p25, p26, // joystick 1 up, down, left, right
        p21, p22, p27, p28, // joystick 2 up, down, left, right
        p29, p30            // button 1, button 2
      );
    }
    
    /*****************************/
    /* READ ALL THE PANEL INPUTS */
    /*****************************/
    // Returns a bitmap of all the inputs.
    void Read( void ) {
      if( bus == (BusIn*)NULL ) {
        inputs = 0;
      }
      else {
        // Note that a closed contact registers as a zero
        // which is why the bitwise not operator (~) is used.
        inputs = ~*bus & 0x3FF;
      }
    }
    
    // Masks to use with GetInputs method.
    enum Mask {
      Up1 = 1,
      Down1 = 2,
      Left1 = 4,
      Right1 = 8,
      Up2 = 16,
      Down2 = 32,
      Left2 = 64,
      Right2 = 128,
      Button1 = 256,
      Button2 = 512,
    };
    
    // Bit numbers indicating where readings for each joystick start in the map.
    // Can use these with right shift operator.
    enum Bits {
      Joy1 = 0,      // joystick 1 readings start at bit 0
      Joy2 = 4,      // joystick 2 readings start at bit 4
      Buttons = 8,   // button readings start at bit 8
    };
    
    /***************************/
    /* GET STATE OF THE INPUTS */
    /***************************/
    // Returns a bitmap of the state of the inputs
    // last time Read was called.
    UInt16 GetInputs( void ) {
      return inputs;
    }
    
  private :
  
    // Stored state of inputs.
    UInt16 inputs;

    // Input bus used to read inputs.
    BusIn *bus;
        
  };
    
#endif

/* END of PanelControls.h */
