/*
 * SOURCE FILE : LevelCollection.cpp
 *
 * Definition of class LevelCollection.
 *
 */

#include <stdio.h>
#include "LevelCollection.h"
#include "Level0.h"
#include "LevelNormal.h"
#include "LevelDescriptor.h"

/***************/
/* CONSTRUCTOR */
/***************/
LevelCollection::LevelCollection() {
}

/**************/
/* DESTRUCTOR */
/**************/
LevelCollection::~LevelCollection() {
}

#define LEVELCOUNT 21

/************************/
/* GET NUMBER OF LEVELS */
/************************/
UInt8 LevelCollection::GetLevelCount( void ) const {
  return LEVELCOUNT;
}

// Level 0 which is NOT dynamically allocated.
// Not a real level. This is attract mode.
static Level0 level0;

// Data for enemies on each level.
static const UInt8 e0 [] = { ENDDESCRIPTOR };
static const UInt8 e1 [] = { Grunt, 5, Crusher, 5, ENDDESCRIPTOR };
static const UInt8 e2 [] = { Grunt, 10, Brain, 2, ENDDESCRIPTOR };
static const UInt8 e3 [] = { Grunt, 10, Crusher, 8, ENDDESCRIPTOR };
static const UInt8 e4 [] = { Grunt, 12, Crusher, 6, Brain, 2, ENDDESCRIPTOR };
static const UInt8 e5 [] = { Grunt, 12, Brain, 4, BlueMeany, 2, ENDDESCRIPTOR };
static const UInt8 e6 [] = { Grunt, 5, Crusher, 8, Brain, 1, BlueMeany, 3, ENDDESCRIPTOR };
static const UInt8 e7 [] = { Brain, 15, ENDDESCRIPTOR };
static const UInt8 e8 [] = { Grunt, 3, Brain, 2, BlueMeany, 5, ENDDESCRIPTOR };
static const UInt8 e9 [] = { Grunt, 1, BlueMeany, 10, ENDDESCRIPTOR };
static const UInt8 e10 [] = { Brain, 5, BlueMeany, 10, ENDDESCRIPTOR };
static const UInt8 e11 [] = { Grunt, 5, Crusher, 5, Brain, 10, ENDDESCRIPTOR };
static const UInt8 e12 [] = { Grunt, 10, Brain, 5, BlueMeany, 9, ENDDESCRIPTOR };
static const UInt8 e13 [] = { BlueMeany, 15, ENDDESCRIPTOR };
static const UInt8 e14 [] = { Brain, 20, ENDDESCRIPTOR };
static const UInt8 e15 [] = { Grunt, 32, BlueMeany, 1, ENDDESCRIPTOR };
static const UInt8 e16 [] = { Grunt, 10, Crusher, 30, ENDDESCRIPTOR };
static const UInt8 e17 [] = { Brain, 25, ENDDESCRIPTOR };
static const UInt8 e18 [] = { Grunt, 16, Brain, 16, ENDDESCRIPTOR };
static const UInt8 e19 [] = { Grunt, 16, Brain, 16, BlueMeany, 5, ENDDESCRIPTOR };
static const UInt8 e20 [] = { Brain, 15, BlueMeany, 15, ENDDESCRIPTOR };

// Array containing level data for each level.
// First parameter is number of humans.
// Second parameter points to an array describing enemies on the level.
static const LevelDescriptor levelDescriptors[ LEVELCOUNT ] = {
    LevelDescriptor( 0, e0 ),
    LevelDescriptor( 5, e1 ),
    LevelDescriptor( 6, e2 ),
    LevelDescriptor( 5, e3 ),
    LevelDescriptor( 6, e4 ),
    LevelDescriptor( 8, e5 ),
    LevelDescriptor( 8, e6 ),
    LevelDescriptor( 8, e7 ),
    LevelDescriptor( 8, e8 ),
    LevelDescriptor( 8, e9 ),
    LevelDescriptor( 8, e10 ),
    LevelDescriptor( 8, e11 ),
    LevelDescriptor( 8, e12 ),
    LevelDescriptor( 12, e13 ),
    LevelDescriptor( 8, e14 ),
    LevelDescriptor( 16, e15 ),
    LevelDescriptor( 16, e16 ),
    LevelDescriptor( 16, e17 ),
    LevelDescriptor( 16, e18 ),
    LevelDescriptor( 16, e19 ),
    LevelDescriptor( 24, e20 ),
};

/***************/
/* GET A LEVEL */
/***************/
// Pass level number in levelNumber.
// Returns pointer to level or NULL if no such level.
Level *LevelCollection::GetLevel( UInt8 levelNumber ) {
  if( levelNumber >= LEVELCOUNT ) {
    // Level number out of range.
    return (Level*)NULL;
  }
  else if( levelNumber == 0 ) {
    // Level zero is the attract mode and is not dynamically allocated.
    return &level0;
  }
  else {
    // Fetch level descriptor for this level and create a Level from it.
    Level *level = new LevelNormal( levelDescriptors + levelNumber );
    // Tag it with the correct level number before returning it.
    level->LevelNumber = levelNumber;
    return level;
  }
}

/*******************************/
/* FREE MEMORY USED BY A LEVEL */
/*******************************/
// Pass pointer to a level.
// Frees memory used by level.
void LevelCollection::FreeLevel( Level *level ) {
    if( ( level != (Level*)NULL ) && level->IsDynamicallyAllocated ) {
        delete level;
    }
}
