/*
 * SOURCE FILE : HumanObject.cpp
 *
 * Represents a wandering human.
 *
 */

#include "HumanObject.h"
#include "Gameduino.h"
#include "FrameCounter.h"
#include "ArenaConst.h"
#include "Animations.h"
#include "Walker.h"

// Index of next animation to use when constructing.
UInt8 HumanObject::nextAnimationIndex = 0;

/***************/
/* CONSTRUCTOR */
/***************/
// Pass number of sprite to use in sn.
// Pass address of animation data in ad.
// Animation data must consist of AnimationStages bytes which gives the sprite
// image numbers used for each state of the animation.
HumanObject::HumanObject() :
  CurrentState( WalkingAbout ),
  animationData( Animations::HumanAnimations[ nextAnimationIndex++ ] ),
  countdown( 100 )
{
  // Wrap around animation index.
  nextAnimationIndex %= Animations::HumanAnimationCount;
  // Initialise horizontal and vertical speeds.
    Walker::InitialiseVelocities( &hSpeed, &vSpeed );
  // Movement is always restricted (property of GameObject).
  MovementRestricted = true;
  // Restrict to boundary of arena.
  Bounds = &ArenaRectangle;
}

/**************/
/* DESTRUCTOR */
/**************/
HumanObject::~HumanObject() {
}

/*****************************/
/* GET TYPE OF HUMAN THIS IS */
/*****************************/
// Returns type of human.
HumanObject::HumanType HumanObject::GetHumanType( void ) {
    return ( animationData == Animations::WomanAnimation ) ? Woman : Man;
}

/************************/
/* MOVE THE GAME OBJECT */
/************************/
void HumanObject::ProtectedMove( void ) {
  switch( CurrentState ) {
  case WalkingAbout :
    // Make human bounce of edges of restriction area.
        Walker::Walk( &Xco, &Yco, &hSpeed, &vSpeed, RestrictionFlags );
    break;
  case Rescued :
    case Dead :
    // Count down and when count reaches zero make human invisible.
    if( countdown > 0 ) {
      countdown--;
    }
    Visible = ( countdown > 0 );
    break;
  }
}

/************************/
/* DRAW THE GAME OBJECT */
/************************/
// Pass pointer to a Gameduino to draw on in gd.
// This is only called after it has been established that the
// game object is visible.
void HumanObject::Draw( Gameduino *gd ) {
  switch( CurrentState ) {
  case WalkingAbout :
    Walker::Draw( gd, SpriteNumber, Xco, Yco, hSpeed, FrameCounter >> 3, animationData );
    break;
  case Rescued :
    gd->sprite( SpriteNumber, ToPixel( Xco ), ToPixel( Yco ), FiftyImage, 0, Gameduino::None, GoodGuy );
    break;
  case Dead :
    gd->sprite( SpriteNumber, ToPixel( Xco ), ToPixel( Yco ), SkullImage, 0, Gameduino::None, GoodGuy );
    break;
  }
}
