/*
 * SOURCE FILE : CrusherObject.cpp
 *
 * Represents a big square robot that crushes humans.
 *
 */

#include "CrusherObject.h"
#include "Gameduino.h"
#include "FrameCounter.h"
#include "ArenaConst.h"
#include "Animations.h"
#include "Walker.h"

/***************/
/* CONSTRUCTOR */
/***************/
CrusherObject::CrusherObject() :
  animationData( Animations::CrusherAnimation ),
    stunCountdown( 0 ),
    stunFrameCount( 0 )
{
  // Initialise horizontal and vertical speeds.
    Walker::InitialiseVelocities( &hSpeed, &vSpeed );
  // Movement is always restricted (property of GameObject).
  MovementRestricted = true;
  // Restrict to boundary of arena.
  Bounds = &ArenaRectangle;
    // Crushers are indestructable.
    HitPoints = Indestructable;
    // Crushers squash humans.
    SquashesHumans = true;
}

/**************/
/* DESTRUCTOR */
/**************/
CrusherObject::~CrusherObject() {
}

/************************/
/* MOVE THE GAME OBJECT */
/************************/
void CrusherObject::ProtectedMove( void ) {
    if( stunCountdown > 0 ) {
        stunCountdown--;
    }
    else {    
        Walker::Walk( &Xco, &Yco, &hSpeed, &vSpeed, RestrictionFlags );
    }
}

/************************/
/* DRAW THE GAME OBJECT */
/************************/
// Pass pointer to Gameduino to draw on in gd.
// This is only called after it has been established that the
// game object is visible.
void CrusherObject::Draw( Gameduino *gd ) {
    // Only update stunFrameCount when not stunned.
    // This halts animation when in stunned state.
    if( stunCountdown == 0 ) {
        stunFrameCount = FrameCounter >> 3;
    }
    Walker::Draw( gd, SpriteNumber, Xco, Yco, hSpeed, stunFrameCount, animationData );
}

/********************************************/
/* INFORM ENEMY IT HAS BEEN HIT BY A BULLET */
/********************************************/
// Default implementation does nothing but if special behaviour
// is required in a derived class then this should be overridden.
// Note that this does NOT deal with determining if the enemy is dead or not.
// An enemy ALWAYS dies if HitPoints reaches zero.
void CrusherObject::RegisterHitByBullet( void ) {
    stunCountdown = 10;
}
