/*
 * SOURCE FILE : CharacterNet.cpp
 *
 */
 
 #include "CharacterNet.h"
 
// An instance of this class.
CharacterNet CharacterNet::instance;

// Default character codes to use.
const UInt8 CharacterNet::defaultCodes[ CodePosCount ] = {
    148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158,
};
        
/** Constructor.
 */
CharacterNet::CharacterNet() :
    characterCodes( defaultCodes )
{
}

/** Draw a net of characters.
 * @param gd The Gameduino to draw on.
 * @param xco X coordinate for top left.
 * @param yco Y coordinate for top left.
 * @param columnCount The number of columns to draw.
 * @param rowCount The number of rows to draw.
 * @param cellWidth The width of each cell in characters.
 * @param cellHeight The height of each cell in characters.
 */
void CharacterNet::Draw( Gameduino *gd, UInt8 xco, UInt8 yco, UInt8 columnCount, UInt8 rowCount, UInt8 cellWidth, UInt8 cellHeight ) {
    UInt8 x = xco;
    for( UInt8 column = 0; column < columnCount; ++column ) {
        UInt8 y = yco;
        for( UInt8 row = 0; row < rowCount; ++row ) {
            DrawCell( gd, x, y, cellWidth, cellHeight, ( row == 0 ), ( column == 0 ), ( row == rowCount - 1 ), ( column == columnCount - 1 ) );
            y += cellHeight;
        }
        x += cellWidth;
    }
}

/** Draw a single cell.
 * Note that only the upper and left hand edges are drawn.
 * @param gd Gameduino to draw on.
 * @param xco X coordinate for top left.
 * @param yco Y coordinate for top left.
 * @param cellWidth Width of cell in characters.
 * @param cellHeight Height of cell in characters.
 * @param topRow Set to true if this is a cell in the top row of the net.
 * @param leftColumn Set to true if this is a cell in the leftmost column of the net.
 * @param bottomRow Set to true if this is a cell in the bottom row of the net.
 * @param rightColumn Set to true if this is a cell in the rightmost column of the net.
 */
void CharacterNet::DrawCell( Gameduino *gd, UInt8 xco, UInt8 yco, UInt8 cellWidth, UInt8 cellHeight, bool topRow, bool leftColumn, bool bottomRow, bool rightColumn ) {
    // Draw character at top left corner of cell.
    char c;
    if( topRow ) {
        if( leftColumn ) {
            c = characterCodes[ TopLeftCorner ];
        }
        else {
            c = characterCodes[ TDown ];
        }
    }
    else if( leftColumn ) {
        c = characterCodes[ TRight ];
    }
    else {
        c = characterCodes[ Cross ];
    }
    gd->putchar( xco, yco, c );
    // If in the rightmost column then draw character at top right of cell.
    if( rightColumn ) {
        c = ( topRow ? characterCodes[ TopRightCorner ] : characterCodes[ TLeft ] );
        gd->putchar( xco + cellWidth, yco, c );
    }
    // If in the bottom row then draw character at bottom left of cell.
    if( bottomRow ) {
        c = ( leftColumn ? characterCodes[ BottomLeftCorner ] : characterCodes[ TUp ] );
        gd->putchar( xco, yco + cellHeight, c );
    }
    // If in the rightmost column and the bottom row then draw character at bottom right of cell.
    if( rightColumn && bottomRow ) {
        gd->putchar( xco + cellWidth, yco + cellHeight, characterCodes[ BottomRightCorner ] );
    }
    // Draw top edge.
    c = characterCodes[ Horizontal ];
    for( UInt8 x = xco + 1; x < xco + cellWidth; ++x ) {
        gd->putchar( x, yco, c );
    }
    // If in bottom row then draw bottom edge.
    if( bottomRow ) {
        for( UInt8 x = xco + 1; x < xco + cellWidth; ++x ) {
            gd->putchar( x, yco + cellHeight, c );
        }
    }
    // Draw left edge.
    c = characterCodes[ Vertical ];
    for( UInt8 y = yco + 1; y < yco + cellHeight; ++y ) {
        gd->putchar( xco, y, c );
    }
    // If in rightmost column then draw right edge.
    if( rightColumn ) {
        for( UInt8 y = yco + 1; y < yco + cellHeight; ++y ) {
            gd->putchar( xco + cellWidth, y, c );
        }
    }
}
