/*
 * SOURCE FILE : CharacterNet.h
 *
 */
 
 #ifndef CharacterNetIncluded
 
    #define CharacterNetIncluded
 
    #include "Gameduino.h"
    
    /** Used to draw net like grids of characters on a Gameduino display.
     * Requires the Gameduino library.
     */
    class CharacterNet {
    
    public :
    
        /** Constructor.
         */
        CharacterNet();
        
        /** Enumeration defining order of codes in array passed to SetCharacterCodes.
         */
        enum CodePos {
            TopLeftCorner,
            TopRightCorner,
            BottomLeftCorner,
            BottomRightCorner,
            Horizontal,
            Vertical,
            TDown,
            TUp,
            TLeft,
            TRight,
            Cross,
            CodePosCount            // number of codes, MUST COME LAST
        };
        
        /** Set character codes to use when drawing.
         * Array must contain CodePosCount items and order of
         * codes is defined by CodePos enumeration.
         * @param codes Array of characters codes to use.
         */
        void SetCharacterCodes( const UInt8 *codes ) {
            characterCodes = codes;
        }

        /** Get character codes currently in use when drawing.
         * @param codes Array of characters codes in use.
         */
        const UInt8 *GetCharacterCodes( void ) const {
            return characterCodes;
        }
                 
        /** Draw a net of characters.
         * @param gd The Gameduino to draw on.
         * @param xco X coordinate for top left.
         * @param yco Y coordinate for top left.
         * @param columnCount The number of columns to draw.
         * @param rowCount The number of rows to draw.
         * @param cellWidth The width of each cell in characters.
         * @param cellHeight The height of each cell in characters.
         */
        void Draw( Gameduino *gd, UInt8 xco, UInt8 yco, UInt8 columnCount, UInt8 rowCount, UInt8 cellWidth, UInt8 cellHeight );
        
        /** Draw a single cell.
         * Note that only the upper and left hand edges are drawn.
         * @param gd Gameduino to draw on.
         * @param xco X coordinate for top left.
         * @param yco Y coordinate for top left.
         * @param cellWidth Width of cell in characters.
         * @param cellHeight Height of cell in characters.
         * @param topRow Set to true if this is a cell in the top row of the net.
         * @param leftColumn Set to true if this is a cell in the leftmost column of the net.
         * @param bottomRow Set to true if this is a cell in the bottom row of the net.
         * @param rightColumn Set to true if this is a cell in the rightmost column of the net.
         */
        void DrawCell( Gameduino *gd, UInt8 xco, UInt8 yco, UInt8 cellWidth, UInt8 cellHeight, bool topRow, bool leftColumn, bool bottomRow, bool rightColumn );

        /** Get an instance of this class.
         * @param Returns a pointer to a static instance of this class.
         */
        static CharacterNet *GetInstance( void ) {
            return &instance;
        }
         
    private :

        // An instance of this class.
        static CharacterNet instance;

        // Default character codes to use.
        static const UInt8 defaultCodes[ CodePosCount ];

        // Pointer to character codes currently in use.
        const UInt8 *characterCodes;

    };
       
 #endif
 
 /* END of CharacterNet.h */
 