/*
 * SOURCE FILE : BarIndicator.h
 *
 * Definition of class BarIndicator.
 *
 */

#ifndef BarIndicatorDefined

  #define BarIndicatorDefined

  #include "Gameduino.h"
  
  /** Used to draw moving bar indicators on a Gameduino display.
   * The indicators are constructed using suitably defined characters.
   */
  class BarIndicator {

  public :

    /** Constructor.
     */
    BarIndicator();

    /** Destructor.
     */
    virtual ~BarIndicator();

    /** Set character code at which thermometer characters start.
     * There should be 10 characters defined starting at this code, in this order:
     * 0 : Empty
     * 1 : 1/8 full
     * 2 : 1/4 full
     * 3 : 3/8 full
     * 4 : 1/2 full
     * 5 : 5/8 full
     * 6 : 3/4 full
     * 7 : 7/8 full
     * 8 : Full
     * 9 : Greyed out (used when reading is invalid).
     * @param sc Character code at which thermometer characters start.
     */
    void SetStartCode( UInt8 sc ) {
      startCode = sc;
    }
     
    /** Set minimum and maximum readings.
     * @param min Smallest reading that can be shown on bar indicator.
     * @param max Largest reading that can be shown on bar indicator.
     */
    void SetLimits( float min, float max ) {
      this->min = min;
      this->max = max;
    }
    
    /** Set location of indicator on screen using character coordinates.
     * @param x X coordinate for indicator.
     * @param y Y coordinate for indicator.
     */
    void SetLocation( UInt8 x, UInt8 y ) {
      xco = x;
      yco = y;
    }
    
    /** Set length of indicator (in characters).
     * @param len Length of indicator.
     */
    void SetLength( UInt8 len ) {
      length = len;
    }
    
    /** Draw bar indicator.
     * Invalid readings are drawn using a greyed out indicator.
     * @param gd Pointer to Gameduino to use for drawing.
     * @param reading Reading to display on indicator.
     * @param valid True if reading is valid, False if not.
     */
    void Draw( Gameduino *gd, float reading, bool valid );
    
  private :
  
    // Character code at which thermometer characters start.
    UInt8 startCode;
    
    // Minimum and maximum readings on thermometer.
    float min, max;

    // Coordinates of indicator.
    UInt8 xco, yco;
       
    // Length of indicator.
    UInt8 length;
     
  };

#endif

/* END of BarIndicator.h */
