/*
 * SOURCE FILE : BarIndicator.cpp
 *
 * Definition of class BarIndicator.
 *
 */

#include "BarIndicator.h"

/** Constructor.
 */
BarIndicator::BarIndicator() :
    startCode( 128 ),
    min( (float)0 ),
    max( (float)100 ),
    xco( 0 ),
    yco( 0 ),
    length( 10 )
{
}

/** Destructor.
 */
BarIndicator::~BarIndicator() {
}

/** Draw bar indicator.
 * Invalid readings are drawn using a greyed out indicator.
 * @param gd Pointer to Gameduino to use for drawing.
 * @param reading Reading to display on indicator.
 * @param valid True if reading is valid, False if not.
 */
void BarIndicator::Draw( Gameduino *gd, float reading, bool valid ) {
    char text[] = "X";
    if( valid ) {
        // Work out fraction of indicator link that is filled.
        float frac = ( reading - min ) / ( max - min );
        // Don't allow fraction to stray outside of range 0.0 to 1.0.
        if( frac < (float)0 ) {
          frac = (float)0;
        }
        else if( frac > (float)1 ) {
          frac = (float)1;
        }
        // Work out character position of character that is partially filled.
        UInt8 pos = (UInt8)( frac * (float)length );
        // Draw filled part of indicator.
        UInt8 x;
        text[ 0 ] = (char)( startCode + 8 );
        for( x = xco; x < xco + pos; ++x ) {
          gd->putstr( x, yco, text );
        }
        // Don't do rest if indicator is at maximum.
        if( pos < length ) {
            // Work out how much of next character needs to be filled.
            UInt8 rem = (UInt8)( (int)floor( frac * (float)length * (float)8 ) & 7 );
            // Draw partially filled character.
            text[ 0 ] = startCode + rem;
            gd->putstr( xco + pos, yco, text );
            // Draw unfilled characters for remainder of indicator.
            text[ 0 ] = (char)( startCode + 0 );
            for( x = xco + pos + 1; x < xco + length; ++x ) {
              gd->putstr( x, yco, text );
            }
        }
    }
    else {
        // Draw greyed out characters for whole indicator.
        text[ 0 ] = (char)( startCode + 9 );
        for( UInt8 x = xco; x < xco + length; ++x ) {
          gd->putstr( x, yco, text );
        }
    }
}
